<?php

namespace Asics\Verifone\Plugin\Model;

use Asics\Verifone\Helper\Configuration;
use Asics\Verifone\Model\Config\Backend\MerchantLogo as ConfigMerchantLogo;
use Exception;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Filesystem;
use Magento\Framework\Filesystem\Directory\ReadInterface;
use Magento\Framework\UrlInterface;
use Magento\MediaStorage\Helper\File\Storage\Database;
use Psr\Log\LoggerInterface;
use Magento\Framework\Exception\ValidatorException;
use Verifone\Hosted\Model\VerifoneConfigProvider as VFConfigProvider;

class VerifoneConfigProvider
{
    /**
     * @var string
     */
    protected $_logoPrefix;

    /**
     * Media directory instance
     *
     * @var ReadInterface
     */
    protected $mediaDirectory;

    /**
     * Filesystem instance
     *
     * @var Filesystem
     */
    protected $_filesystem;

    /**
     * @var UrlInterface
     */
    protected $_urlBuilder;
    
    /**
     * @var Configuration
     */
    protected $_configuration;

    /**
     * @var Database
     */
    protected $_fileStorageHelper;

    /**
     * @var LoggerInterface
     */
    protected $logger;

    /**
     * VerifoneConfigProvider constructor.
     *
     * @param Filesystem $filesystem
     * @param UrlInterface $urlBuilder
     * @param Database $fileStorageHelper
     * @param Configuration $configuration
     * @param LoggerInterface $logger
     */
    public function __construct (
        Filesystem $filesystem,
        UrlInterface $urlBuilder,
        Database $fileStorageHelper,
        Configuration $configuration,
        LoggerInterface $logger
    ) {
        $this->_filesystem = $filesystem;
        $this->_urlBuilder = $urlBuilder;
        $this->_fileStorageHelper = $fileStorageHelper;
        $this->_configuration = $configuration;
        $this->logger = $logger;
    }

    /**
     * @param VFConfigProvider $subject
     * @param $result
     * @return array
     */
    public function afterGetConfig (
        VFConfigProvider $subject,
        $result
    ) {
        try {
            $logos = [];
            for ($i = 1; $i <= 5; $i++) {
                $this->setLogoPathPrefix("display_settings/merchantLogo{$i}");
                $url = $this->getLogoUrl();
                if (empty($url)) {
                    continue;
                }
                $logos[] = [
                    "Url" => $url,
                    "Alt" => $this->getLogoAlt(),
                    "Width" => $this->getLogoWidth(),
                    "Height" => $this->getLogoHeight()
                ];
            }

            if (count($logos) > 0 && isset($result['payment']['verifone_hosted'])) {
                $result['payment']['verifone_hosted']['merchantUICustomOptions']['logos'] = $logos;
            }
        } catch (Exception $e) {
            $this->logger->error($e->getMessage());
            $this->logger->error($e->getTraceAsString());
        }

        return $result;
    }

    /**
     * Sets the logo path prefix
     *
     * @param string $logoPrefix
     */
    public function setLogoPathPrefix($logoPrefix)
    {
        $this->_logoPrefix = $logoPrefix;
    }

    /**
     * Get logo image URL
     *
     * @return string
     * @throws ValidatorException
     */
    public function getLogoUrl()
    {
        $folderName = ConfigMerchantLogo::UPLOAD_DIR;
        $storeLogoPath = $this->_configuration->getLogoSource($this->_logoPrefix);
        $path = $folderName . '/' . $storeLogoPath;
        $logoUrl = $this->_urlBuilder->getBaseUrl(
            [
            '_type' => UrlInterface::URL_TYPE_MEDIA
            ]
        ) . $path;

        $url = "";
        if ($storeLogoPath !== null && $this->_isFile($path)) {
            $url = $logoUrl;
        }

        return $url;
    }

    /**
     * Retrieve logo text
     *
     * @return string
     */
    public function getLogoAlt()
    {
        return $this->_configuration->getLogoAlt($this->_logoPrefix);
    }

    /**
     * Retrieve logo width
     *
     * @return int
     */
    public function getLogoWidth()
    {
        return $this->_configuration->getLogoWidth($this->_logoPrefix);
    }

    /**
     * Retrieve logo height
     *
     * @return int
     */
    public function getLogoHeight()
    {
        return $this->_configuration->getLogoHeight($this->_logoPrefix);
    }

    /**
     * If DB file storage is on - find there, otherwise - just file_exists
     *
     * @param $filename
     * @return bool
     */
    protected function _isFile($filename)
    {
        if ($this->_fileStorageHelper->checkDbUsage() && !$this->getMediaDirectory()->isFile($filename)) {
            $this->_fileStorageHelper->saveFileToFilesystem($filename);
        }
        return $this->getMediaDirectory()->isFile($filename);
    }

    /**
     * Get media directory
     *
     * @return ReadInterface
     */
    protected function getMediaDirectory()
    {
        if (!$this->mediaDirectory) {
            $this->mediaDirectory = $this->_filesystem->getDirectoryRead(DirectoryList::MEDIA);
        }
        return $this->mediaDirectory;
    }
}
