<?php
declare(strict_types=1);

namespace BrittainWynyard\OneAsics\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Store\Model\ScopeInterface;

class Config extends AbstractHelper
{

    const XML_PATH_ENABLED = 'oneasics/general/enabled';

    const XML_PATH_MODE= 'oneasics/oauth/mode';
    const XML_PATH_CLIENTID = 'oneasics/oauth/client_id';
    const XML_PATH_CLIENTSECRET = 'oneasics/oauth/client_secret';
    const XML_PATH_STYLE = 'oneasics/oauth/style';
    const XML_PATH_LOCALE = 'oneasics/oauth/locale';
    const XML_PATH_PRIVACYPAGE= 'oneasics/oauth/privacy_page';
    const XML_PATH_TERMSPAGE = 'oneasics/oauth/terms_page';
    const XML_PATH_ONEASICS_TERMSPAGE = 'oneasics/oauth/oneasics_terms_page';

    const XML_PATH_TERMSPRIVACYCOUNTRY = 'oneasics/oauth/terms_privacy_country';
    const XML_PATH_TERMSPRIVACYVERSION = 'oneasics/oauth/terms_privacy_version';

    const XML_PATH_DEBUG = 'oneasics/oauth/debug';

    const USER_INFO_API_END_POINT = 'api/v2/userinfo';
    const ACCESS_TOKEN_API_END_POINT = 'api/v2/token';
    const ONE_ASICS_PRODUCTION_BASE_URL = 'https://id.asics.com/';
    const ONE_ASICS_STAGING_BASE_URL = 'https://id-sandbox.asics.com/';
    const ONE_ASICS_DEV_BASE_URL = 'https://id-dev.asics.com/';
    const CHANGE_PASSWORD_API_END_POINT = 'change-password';
    const EMAIL_API_END_POINT = 'change-email';
    const CHANGE_EMAIL_API_END_POINT = 'api/v1/users/change-email';
    const LANDING_PAGE_API_END_POINT = 'lander';
    const GENDER_MALE = 'male';
    const GENDER_FEMALE = 'female';
    const GENDER_OTHER = 'other';
    const GENDER_BOTH = 'both';
    const INTERESTED_IN_GENDER_MALE = 'mens';
    const INTERESTED_IN_GENDER_FEMALE = 'womens';
    const INTERESTED_IN_GENDER_BOTH = 'both';


    protected $cmsHelper;
    protected $pageFactory;
    protected $storeManager;

    /**
     * @param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(
        \Magento\Framework\App\Helper\Context $context,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        \Magento\Cms\Model\PageFactory $pageFactory,
        \Magento\Cms\Helper\Page $cmsHelper

    ) {
        $this->storeManager = $storeManager;
        $this->pageFactory = $pageFactory;
        $this->cmsHelper = $cmsHelper;

        parent::__construct($context);
    }

    /**
     * @return bool
     */
    public function isEnabled()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_ENABLED,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getMode()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_MODE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getClientId()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_CLIENTID,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getClientSecret()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_CLIENTSECRET,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @param string $path
     * @return string
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getRedirectUri($path = 'oneasics/login')
    {
        return $this->storeManager->getStore()->getBaseUrl() . $path;
    }

    /**
     * @return string|null
     */
    public function getPrivacyUrl()
    {
        $page = $this->pageFactory->create()->load($this->getPrivacyPage());
        if ($page->getId()) {
            return $this->cmsHelper->getPageUrl($page->getId());
        }
    }

    /**
     * @return mixed
     */
    public function getPrivacyPage()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_PRIVACYPAGE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return string|null
     */
    public function getOneAsicsTermsUrl()
    {
        $page = $this->pageFactory->create()->load($this->getOneAsicsTermsPage());
        if ($page->getId()) {
            return $this->cmsHelper->getPageUrl($page->getId());
        }
    }

    /**
     * @return string|null
     */
    public function getTermsUrl()
    {
        $page = $this->pageFactory->create()->load($this->getTermsPage());
        if ($page->getId()) {
            return $this->cmsHelper->getPageUrl($page->getId());
        }
    }

    /**
     * @return mixed
     */
    public function getOneAsicsTermsPage()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_ONEASICS_TERMSPAGE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getTermsPage()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_TERMSPAGE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getLocale()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_LOCALE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getTermsPrivacyCountry()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_TERMSPRIVACYCOUNTRY,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getTermsPrivacyVersion()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_TERMSPRIVACYVERSION,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return mixed
     */
    public function getStyle()
    {
        return $this->scopeConfig->getValue(
            self::XML_PATH_STYLE,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return bool
     */
    public function isDebugMode()
    {
        return (bool)$this->scopeConfig->getValue(
            self::XML_PATH_DEBUG,
            ScopeInterface::SCOPE_STORE
        );
    }

    /**
     * @return string
     */
    public function getBaseUrl()
    {
        switch ($this->getMode()) {
            case 'production':
                return self::ONE_ASICS_PRODUCTION_BASE_URL;
            case "staging":
                return self::ONE_ASICS_STAGING_BASE_URL;
            case "dev":
                return self::ONE_ASICS_DEV_BASE_URL;
        }
    }

    /**
     * Maps gender attribute value to interested_in_gender attribute value.
     * Takes in gender and outputs interested_in_gender (MALE, FEMALE, BOTH)
     * @param string $gender
     * @return string
     */
    public function getInterestedInGenderFromGender(string $gender): string
    {
        $gender_interested_in_gender_to_map = [
            self::GENDER_MALE=> self::INTERESTED_IN_GENDER_MALE,
            self::GENDER_FEMALE=> self::INTERESTED_IN_GENDER_FEMALE,
            self::GENDER_BOTH=> self::INTERESTED_IN_GENDER_BOTH,
        ];

        return array_key_exists($gender, $gender_interested_in_gender_to_map)?
            $gender_interested_in_gender_to_map[$gender] : '';
    }
}

