<?php

namespace Overdose\DataLayer\Helper;

use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Model\ProductRepository;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\App\Helper\Context;
use Anowave\Ec\Helper\Data as EcHelper;

class Ec extends \Magento\Framework\App\Helper\AbstractHelper
{
    /**
     * @var ProductRepository
     */
    protected $productRepository;

    /**
     * @var CategoryRepositoryInterface
     */
    protected $categoryRepository;

    /** @var CategoryCollectionFactory */
    protected $categoryCollectionFactory;

    /**
     * @var EcHelper
     */
    protected $ecHelper;

    /**
     * @var Configurable
     */
    protected $configurable;

    /**
     * @param Context $context
     * @param ProductRepository $productRepository
     * @param CategoryRepositoryInterface $categoryRepository
     * @param CategoryCollectionFactory $categoryCollectionFactory
     * @param EcHelper $ecHelper
     * @param Configurable $configurable
     */
    public function __construct(
        Context                     $context,
        ProductRepository           $productRepository,
        CategoryRepositoryInterface $categoryRepository,
        CategoryCollectionFactory   $categoryCollectionFactory,
        EcHelper                    $ecHelper,
        Configurable                $configurable
    ) {
        $this->productRepository = $productRepository;
        $this->categoryRepository = $categoryRepository;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->ecHelper = $ecHelper;
        $this->configurable = $configurable;
        parent::__construct($context);
    }

    /**
     * @param $product
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getCategoryData($product)
    {
        $result = [];
        $mainCategoryId = $product->getMainCategory();
        if ($mainCategoryId) {
            $mainCategory = $this->categoryRepository->get($mainCategoryId);
            $catCollection = $this->categoryCollectionFactory->create()
                ->addAttributeToSelect(['name', 'url'])
                ->addAttributeToFilter('entity_id', ['in' => $mainCategory->getPathIds()]);
            foreach ($catCollection as $catItem) {
                if ($catItem->getLevel() > 1) {
                    if (!isset($result['item_category'])) {
                        $result['item_category'] = $catItem->getName();
                    } elseif (!isset($result['item_category2'])) {
                        $result['item_category2'] = $catItem->getName();
                    } elseif (!isset($result['item_category3'])) {
                        $result['item_category3'] = $catItem->getName();
                    } elseif (!isset($result['item_category4'])) {
                        $result['item_category4'] = $catItem->getName();
                    }
                }
            }
        }
        return $result;
    }

    /**
     * @param $product
     * @return int
     */
    public function checkSpecialPriceProduct($product)
    {
        if ($product->getTypeId() == 'bundle') {
            $finalPrice = $product->getPriceInfo()->getPrice('final_price')->getMinimalPrice()->getValue();
            $orgPrice = $product->getPriceInfo()->getPrice('regular_price')->getMinimalPrice()->getValue();
        } elseif ($product->getTypeId() == 'grouped') {
            $usedProds = $product->getTypeInstance(true)->getAssociatedProducts($product);
            $finalPrice = 0;
            $orgPrice = 0;
            foreach ($usedProds as $child) {
                if ($child->getId() != $product->getId()) {
                    $finalPrice += $child->getFinalPrice();
                    $orgPrice += $child->getPrice();
                }
            }
        } elseif ($product->getTypeId() == 'configurable') {
            $finalPrice = $product->getPriceInfo()->getPrice('final_price')->getMinimalPrice()->getValue();
            $orgPrice = $product->getPriceInfo()->getPrice('regular_price')->getAmount()->getValue();
        } else {
            $finalPrice = $product->getFinalPrice();
            $orgPrice = $product->getPrice();
        }
        if ($finalPrice && $orgPrice && $finalPrice < $orgPrice) {
            return $orgPrice - $finalPrice;
        }
        return 0;
    }

    /**
     * @param $product
     * @return string
     */
    public function getProductWidth($product)
    {
        try {
            return $product->getResource()->getAttribute('width')->getFrontend()->getValue($product);

        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * @param $product
     * @return mixed
     */
    public function getProductColor($product)
    {
        try {
            $marketingColour = $product->getResource()->getAttribute('mkt_colour')->getFrontend()->getValue($product);
            if (!$marketingColour) {
                $marketingColour = $product->getName();
            }
            return $marketingColour;
        } catch (\Exception $e) {
            return $product->getName();
        }
    }

    /**
     * @param $product
     * @return bool|mixed|string
     */
    public function getAffiliation($product)
    {
        return $this->ecHelper->getBrand($product);
    }

    /**
     * @param $configurableProduct
     * @return array
     */
    public function getDiscountandSizeMapping($configurableProduct)
    {
        $result = [];
        $result['discount_mapping'] = [];
        $result['size_mapping'] = [];

        try {

            if ($configurableProduct->getTypeId() == Configurable::TYPE_CODE) {
                $childProducts = $this->configurable->getUsedProducts($configurableProduct);
                foreach ($childProducts as $childProduct) {
                    $result['discount_mapping'][$childProduct->getSku()] = $this->checkSpecialPriceProduct($childProduct);
                    $size = $childProduct->getResource()->getAttribute('size');
                    if ($size) {
                        $result['size_mapping'][$childProduct->getSku()] = $size->getFrontend()->getValue($childProduct);
                    }
                }
            } else {
                $result['discount_mapping'][$configurableProduct->getSku()] = $this->checkSpecialPriceProduct($configurableProduct);
            }
        } catch (\Exception $e) {
            return $result;
        }
        return $result;
    }

    /**
     * @param $sku
     * @return \Magento\Catalog\Api\Data\ProductInterface|\Magento\Catalog\Model\Product|null
     */
    public function getProductBySku($sku)
    {
        try {
            return $this->productRepository->get($sku);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * @param $product
     * @return string
     */
    public function getSize($product)
    {
        try {
            return $product->getResource()->getAttribute('size')->getFrontend()->getValue($product);
        } catch (\Exception $e) {
            return '';
        }
    }

    /**
     * @param $data
     * @param $product
     * @return mixed|string
     */
    public function getItemListId($data, $product)
    {
        $result = '';
        $productId = $product->getId();
        try {
            if ($data) {
                $decodeData = json_decode($data, true);
                if (isset($decodeData['item_list_id']) && (int)$productId == $decodeData['product_id']) {
                    return $decodeData['item_list_id'];
                }
            }
            $mainCategoryId = $product->getMainCategory();
            if ($mainCategoryId) {
                return $mainCategoryId;
            }
        } catch (\Exception $e) {

        }


        return $result;
    }

    /**
     * @param $data
     * @param $product
     * @return mixed|string
     */
    public function getItemListName($data, $product)
    {
        $result = '';
        $productId = $product->getId();
        try {
            if ($data) {
                $decodeData = json_decode($data, true);
                if (isset($decodeData['item_list_name']) && (int)$productId == $decodeData['product_id']) {
                    return $decodeData['item_list_name'];
                }
            }
            $mainCategoryId = $product->getMainCategory();
            if($mainCategoryId) {
                $mainCategory = $this->categoryRepository->get($mainCategoryId);
                if ($mainCategory && $mainCategory->getId()) {
                    return $mainCategory->getName();
                }
            }
        } catch (\Exception $e) {

        }

        return $result;
    }

    /**
     * @param $data
     * @param $sku
     * @param $productId
     * @return int|mixed
     */
    public function getProductIndex($data, $sku, $productId)
    {
        $result = 0;
        try {
            if ($data) {
                $decodeData = json_decode($data, true);
                if (isset($decodeData['item_list']) && is_array($decodeData['item_list']) && (int)$productId == $decodeData['product_id']) {
                    foreach ($decodeData['item_list'] as $item) {
                        if (strtolower($item['item_id']) == strtolower($sku)) {
                            return $item['index'];
                        }
                    }
                }
            }

        } catch (\Exception $e) {

        }
        return $result;
    }

    /**
     * @param $odAddToCartSession
     * @param $product
     * @return string
     */
    public function getSessionItemListId($odAddToCartSession, $product)
    {
        $productId = $product->getId();
        if ($odAddToCartSession && isset($odAddToCartSession[$productId])) {
           return $odAddToCartSession[$productId]['item_list_id'] ?? "";
        }
        $mainCategoryId = $product->getMainCategory();
        if ($mainCategoryId) {
            return $mainCategoryId;
        }
        return '';
    }

    /**
     * @param $odAddToCartSession
     * @param $product
     * @return string
     */
    public function getSessionItemListName($odAddToCartSession, $product)
    {
        $productId = $product->getId();
        if ($odAddToCartSession && isset($odAddToCartSession[$productId])) {
            return $odAddToCartSession[$productId]['item_list_name'] ?? "";
        }
        $mainCategoryId = $product->getMainCategory();
        if ($mainCategoryId) {
            try {
                $mainCategory = $this->categoryRepository->get($mainCategoryId);
                if ($mainCategory && $mainCategory->getId()) {
                    return $mainCategory->getName();
                }
            } catch (\Exception $e) {

            }
        }
        return '';
    }

    /**
     * @param $id
     * @return \Magento\Catalog\Api\Data\ProductInterface|mixed|null
     */
    public function getProductById($id)
    {
        try {
            return $this->productRepository->getById($id);
        } catch (\Exception $e) {
            return null;
        }
    }

}
