<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\CustomerCustomAttributes\Observer;

use Magento\Customer\Model\Attribute;
use Magento\Customer\Model\AttributeFactory;
use Magento\CustomerCustomAttributes\Model\ResourceModel\Sales\Order as OrderResource;
use Magento\CustomerCustomAttributes\Model\ResourceModel\Sales\Quote as QuoteResource;
use Magento\Framework\Event\ManagerInterface;
use Magento\Framework\ObjectManagerInterface;
use Magento\TestFramework\Helper\Bootstrap;
use PHPUnit\Framework\TestCase;

/**
 * Test for save customer custom attribute observer.
 *
 * @see \Magento\CustomerCustomAttributes\Observer\EnterpriseCustomerAttributeSave
 * @magentoDbIsolation disabled
 */
class EnterpriseCustomerAttributeSaveTest extends TestCase
{
    /** @var ObjectManagerInterface */
    private $objectManager;

    /** @var ManagerInterface */
    private $eventManager;

    /** @var AttributeFactory */
    private $attributeFactory;

    /** @var Attribute */
    private $attribute;

    /** @var QuoteResource */
    private $quoteResource;

    /** @var OrderResource */
    private $orderResource;

    /**
     * @inheritdoc
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->objectManager = Bootstrap::getObjectManager();
        $this->eventManager = $this->objectManager->get(ManagerInterface::class);
        $this->attributeFactory = $this->objectManager->get(AttributeFactory::class);
        $this->quoteResource = $this->objectManager->get(QuoteResource::class);
        $this->orderResource = $this->objectManager->get(OrderResource::class);
    }

    /**
     * @inheritdoc
     */
    protected function tearDown(): void
    {
        $this->quoteResource->deleteAttribute($this->attribute);
        $this->orderResource->deleteAttribute($this->attribute);

        parent::tearDown();
    }

    /**
     * @return void
     */
    public function testExecute(): void
    {
        $this->attribute = $this->attributeFactory->create();
        $this->attribute->setData(['attribute_code' => 'test_attribute', 'backend_type' => 'varchar']);
        $this->eventManager->dispatch(
            'magento_customercustomattributes_attribute_save',
            ['attribute' => $this->attribute]
        );
        $columnInQuoteTable = $this->quoteResource->getConnection()
            ->tableColumnExists($this->quoteResource->getMainTable(), 'customer_test_attribute');
        $this->assertTrue(
            $columnInQuoteTable,
            sprintf('Column for new attribute not exist in "%s" table.', $this->quoteResource->getMainTable())
        );
        $columnInOrderTable = $this->orderResource->getConnection()
            ->tableColumnExists($this->orderResource->getMainTable(), 'customer_test_attribute');
        $this->assertTrue(
            $columnInOrderTable,
            sprintf('Column for new attribute not exist in "%s" table.', $this->orderResource->getMainTable())
        );
    }
}
