<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-seo
 * @version   2.10.12
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);


namespace Mirasvit\SeoAi\Console\Command;


use Magento\Framework\App\StateFactory;
use Mirasvit\Core\Service\CompatibilityService;
use Mirasvit\SeoAi\Model\ConfigProvider;
use Mirasvit\SeoAi\Service\MetaAutoFixService;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Question\ConfirmationQuestion;

class SeoAuditMetaFixCommand extends \Symfony\Component\Console\Command\Command
{
    private $configProvider;

    private $appStateFactory;

    private $service;

    public function __construct(
        ConfigProvider $configProvider,
        StateFactory $appStateFactory,
        MetaAutoFixService $service
    ) {
        $this->configProvider  = $configProvider;
        $this->appStateFactory = $appStateFactory;
        $this->service         = $service;

        parent::__construct();
    }

    protected function configure()
    {
        $this->setName('mirasvit:seo-ai:fix-audit-meta')
            ->setDescription('Run SEO AI helper to fix meta fields issues reported in SEO audit');

        $this->addOption(
            'reset',
            null,
            InputOption::VALUE_NONE,
            'Remove all SEO rewrites generated by SEO AI helper'
        );
        $this->addOption(
            'limit',
            null,
            InputOption::VALUE_REQUIRED,
            'Set the number of URLs to process. Default limit is 1000'
        );
        $this->addOption(
            'time-limit',
            null,
            InputOption::VALUE_REQUIRED,
            'Set the time limit in seconds. Max time limit is 3 hours'
        );

        parent::configure();
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $this->appStateFactory->create()->setAreaCode('frontend');
        } catch (\Exception $e) {
        }

        if ($input->getOption('reset')) {
            $q = '<info>All SEO rewrites generated by the SEO AI Helper will be removed.</info> Proceed? [y/n]: ';

            /** @var ConfirmationQuestion $question */
            $question = CompatibilityService::getObjectManager()->create(
                ConfirmationQuestion::class,
                [
                    'question'=> $q,
                    'default' => FALSE
                ]
            );

            $helper = $this->getHelper('question');
            $answer = $helper->ask($input, $output, $question);

            if ($answer == 'y') {
                $output->writeln('<info>Removing rewrites</info>...');
                $this->service->reset();
            } else {
                $output->writeln('<info>Aborting reset</info>');
            }

            $output->writeln('<info>Done</info>');

            return 0;
        }

        $cantRunReasons = $this->configProvider->getCantRunReasons();

        if (count($cantRunReasons)) {
            $output->writeln('<error>Can\'t start process.</error> Reasons:');

            foreach ($cantRunReasons as $reason) {
                $output->writeln('- ' . $reason);
            }

            return 0;
        }

        $limit     = $input->getOption('limit') ? (int)$input->getOption('limit') : 1000;
        $timeLimit = (int)$input->getOption('time-limit');

        if ($timeLimit > 3*60*60) {
            $timeLimit = 3*60*60; // 3 hours max execution time for command
        }

        $output->writeln('<info>Generating SEO rewrites.</info> URLs limit: ' . $limit . '; Time limit (seconds): ' . $timeLimit);

        $this->service->processUrls($limit, $timeLimit, $output);

        $output->writeln('<info>Done</info>');

        return 0;
    }
}
