<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-seo
 * @version   2.10.12
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */


declare(strict_types=1);

namespace Mirasvit\SeoMarkup\Block\Rs;

use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ResourceModel\Product\Collection as ProductCollection;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Registry;
use Magento\Framework\View\Element\Template;
use Magento\Framework\View\Element\Template\Context;
use Magento\Store\Model\Store;
use Mirasvit\Core\Service\SerializeService;
use Mirasvit\Seo\Api\Service\TemplateEngineServiceInterface;
use Mirasvit\SeoContent\Api\Data\TemplateInterface;
use Mirasvit\SeoMarkup\Model\Config;
use Mirasvit\SeoMarkup\Model\Config\CategoryConfig;
use Mirasvit\SeoMarkup\Service\ProductRichSnippetsService;

class Category extends Template
{
    private $category;

    protected $categoryConfig;

    protected $productCollectionFactory;

    private $templateEngineService;

    private $registry;

    protected $productSnippetService;

    protected $productCollection;

    private $storeId;

    public function __construct(
        CategoryConfig                 $categoryConfig,
        ProductCollectionFactory       $productCollectionFactory,
        TemplateEngineServiceInterface $templateEngineService,
        Registry                       $registry,
        Context                        $context,
        ProductRichSnippetsService     $productSnippetService
    ) {
        $this->categoryConfig           = $categoryConfig;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->templateEngineService    = $templateEngineService;
        $this->registry                 = $registry;
        $this->productSnippetService    = $productSnippetService;

        parent::__construct($context);
    }

    protected function _toHtml(): string
    {
        $data = $this->getJsonData();

        if (!$data) {
            return '';
        }

        return '<script type="application/ld+json">' . SerializeService::encode($data) . '</script>';
    }

    public function getJsonData(): ?array
    {
        $this->category = $this->registry->registry('current_category');

        if (!$this->category) {
            return null;
        }

        if ($this->category->getId() == $this->_storeManager->getStore()->getRootCategoryId()) {
            return null;
        }

        if (!$this->categoryConfig->isRsEnabled($this->getStoreId())) {
            return null;
        }

        return $this->getResult();
    }

    protected function getDataAsWebPage(): array
    {
        $url = $this->_urlBuilder->escape($this->_urlBuilder->getCurrentUrl());

        $name = $this->category->getName();

        $image = $this->getImage($this->category);

        $description = $this->getDescription($this->category);

        $result = [
            '@context' => Config::HTTP_SCHEMA_ORG,
            '@type'    => 'WebPage',
            'url'      => $url,
        ];

        if ($this->categoryConfig->getProductOffersType($this->getStoreId())) {
            $itemList = $this->getItemList();

            $result['mainEntity'] = [
                '@type'           => 'offerCatalog',
                'name'            => $name,
                'url'             => $url,
                'numberOfItems'   => count($itemList) ?: '',
                'itemListElement' => $itemList,
            ];

            if (!empty($image)) {
                $result['mainEntity']['image'] = $image;
            }

            if (!empty($description)) {
                $result['mainEntity']['description'] = $description;
            }
        } else {
            $result['name'] = $name;

            if (!empty($image)) {
                $result['image'] = $image;
            }

            if (!empty($description)) {
                $result['description'] = $description;
            }
        }

        return $result;
    }

    protected function getDataAsItemList(): array
    {
        $itemList = [];
        $position = 1;

        foreach ($this->getProductCollection() as $product) {
            $itemList[] = $this->productSnippetService->getListItemJsonData($product, $position++, true);
        }

        $result = [
            '@context'        => Config::HTTP_SCHEMA_ORG,
            '@type'           => 'ItemList',
            'name'            => $this->category->getName(),
            'url'             => $this->_urlBuilder->escape($this->_urlBuilder->getCurrentUrl()),
            'numberOfItems'   => count($itemList),
            'itemListElement' => $itemList,
        ];

        $image = $this->getImage($this->category);

        if (!empty($image)) {
            $result['image'] = $image;
        }

        $description = $this->getDescription($this->category);

        if (!empty($description)) {
            $result['description'] = $description;
        }

        return $result;
    }

    protected function getProductCollection(): ?ProductCollection
    {
        if ($this->productCollection === null) {
            $productOffersType = $this->categoryConfig->getProductOffersType($this->getStoreId());
            switch ($productOffersType) {
                case (CategoryConfig::PRODUCT_OFFERS_TYPE_CURRENT_PAGE):
                    $categoryProductsListBlock = $this->getLayout()->getBlock('category.products.list');

                    if ($categoryProductsListBlock) {
                        $loadedCollection = $categoryProductsListBlock->getLoadedProductCollection();
                        $collection = clone $loadedCollection;
                        $ids = [];

                        foreach ($collection as $product) {
                            $ids[] = $product->getId();
                        }

                        $collection->addAttributeToSelect('*');
                        $collection->addAttributeToFilter(
                            'entity_id',
                            ['in' => $ids]
                        );
                        $collection->addFinalPrice();
                        $collection->load();

                        $this->productCollection = $collection;
                    }
                    break;

                case (CategoryConfig::PRODUCT_OFFERS_TYPE_CURRENT_CATEGORY):
                    $collection = $this->productCollectionFactory->create();
                    $collection->addAttributeToSelect('*');
                    $collection->addCategoryFilter($this->category);
                    $collection->addAttributeToFilter(
                        'visibility',
                        Visibility::VISIBILITY_BOTH
                    );
                    $collection->addAttributeToFilter(
                        'status',
                        Status::STATUS_ENABLED
                    );
                    $collection->addFinalPrice();
                    $collection->load();

                    $this->productCollection = $collection;
                    break;
            }
        }

        return $this->productCollection;
    }

    protected function getItemList(): array
    {
        $data = [];

        if ($this->getProductCollection()) {
            foreach ($this->getProductCollection() as $product) {
                $data[] = $this->productSnippetService->getJsonData($product, true);
            }
        }

        return $data;
    }

    private function getImage(\Magento\Catalog\Model\Category $category): string
    {
        $imageUrl = $category->getImageUrl();

        if ($this->categoryConfig->isImageEnabled($this->getStoreId()) && $imageUrl) {
            return $this->_urlBuilder->escape(
                $this->_storeManager->getStore()->getBaseUrl() . ltrim($imageUrl, '/')
            );
        }

        return '';
    }

    private function getDescription(\Magento\Catalog\Model\Category $category): string
    {
        $value          = '';
        $objectManager  = ObjectManager::getInstance();
        $contentService = $objectManager->get('Mirasvit\SeoContent\Service\ContentService');
        $content        = $contentService->getCurrentContent(TemplateInterface::RULE_TYPE_CATEGORY);

        switch ($this->categoryConfig->getDescriptionType($this->getStoreId())) {
            case CategoryConfig::DESCRIPTION_TYPE_DESCRIPTION:
                $description = $content->getData('category_description');
                $value       = !empty($description)
                    ? $description
                    : $this->templateEngineService->render('[category_description]', ['category' => $category]);
                break;
            case CategoryConfig::DESCRIPTION_TYPE_META:
                $description = $content->getData('meta_description');
                $value       = !empty($description)
                    ? $description
                    : $this->templateEngineService->render('[category_meta_description]', ['category' => $category]);
                break;
        }

        if (!empty($value)) {
            $value = preg_replace('/<style[^>]*>[\s\S]*?<\/style>/i', '', $value);
            $value = trim(strip_tags(str_replace('<', ' <', $value)));
            $value = preg_replace('/\s+/', ' ', $value);
        }

        return $value;
    }

    protected function getStoreId(): int
    {
        if (!isset($this->storeId)) {
            try {
                $this->storeId = (int)$this->_storeManager->getStore()->getId();
            } catch (NoSuchEntityException $exception) {
                return Store::DEFAULT_STORE_ID;
            }
        }

        return $this->storeId;
    }

    protected function getResult(): array
    {
        $productOffersFormat = $this->categoryConfig->getProductOffersFormat($this->getStoreId());

        if ($this->categoryConfig->getProductOffersType($this->getStoreId())
            && ($productOffersFormat == CategoryConfig::PRODUCT_OFFERS_FORMAT_SIMPLE)
            && $this->getProductCollection() && $this->getProductCollection()->getSize()
        ) {
            $result = $this->getDataAsItemList();
        } else {
            $result = $this->getDataAsWebPage();
        }

        return $result;
    }
}
