<?php
/**
 * Mirasvit
 *
 * This source file is subject to the Mirasvit Software License, which is available at https://mirasvit.com/license/.
 * Do not edit or add to this file if you wish to upgrade the to newer versions in the future.
 * If you wish to customize this module for your needs.
 * Please refer to http://www.magentocommerce.com for more information.
 *
 * @category  Mirasvit
 * @package   mirasvit/module-seo
 * @version   2.10.12
 * @copyright Copyright (C) 2025 Mirasvit (https://mirasvit.com/)
 */



namespace Mirasvit\SeoSitemap\Repository\Provider\Mirasvit;

use Magento\Framework\DataObject;
use Magento\Framework\ObjectManagerInterface;
use Magento\Sitemap\Helper\Data as DataHelper;
use Mirasvit\Seo\Service\Alternate\BlogStrategy;
use Mirasvit\Seo\Service\Config\AlternateConfig;
use Mirasvit\SeoSitemap\Api\Repository\ProviderInterface;

class BlogProvider implements ProviderInterface
{
    private $dataHelper;

    private $objectManager;

    private $blogStrategy;

    private $alternateConfig;

    public function __construct(
        ObjectManagerInterface $objectManager,
        DataHelper $sitemapData,
        BlogStrategy $blogStrategy,
        AlternateConfig $alternateConfig
    ) {
        $this->objectManager   = $objectManager;
        $this->dataHelper      = $sitemapData;
        $this->blogStrategy    = $blogStrategy;
        $this->alternateConfig = $alternateConfig;
    }

    /**
     * @return string
     */
    public function getModuleName()
    {
        return 'Mirasvit_BlogMx';
    }

    /**
     * @return bool
     */
    public function isApplicable()
    {
        return true;
    }

    /**
     * @return \Magento\Framework\Phrase|string
     */
    public function getTitle()
    {
        return __('Blog');
    }

    /**
     * @param int $storeId
     *
     * @return array
     */
    public function initSitemapItem($storeId)
    {
        $result = [];

        $result[] = new DataObject([
            'changefreq' => $this->dataHelper->getPageChangefreq($storeId),
            'priority'   => $this->dataHelper->getPagePriority($storeId),
            'collection' => $this->getItems($storeId),
        ]);

        return $result;
    }

    /**
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     *
     * @param int $storeId
     *
     * @return array
     */
    public function getItems($storeId)
    {
        $postRepository = $this->objectManager->get('Mirasvit\BlogMx\Repository\PostRepository');

        $collection = $postRepository->getCollection();
        $collection->addStoreFilter($storeId)
            ->addVisibilityFilter();
        $urlBuilder = class_exists('Mirasvit\BlogMx\Model\Url\UrlBuilder') ? $this->objectManager->get('Mirasvit\BlogMx\Model\Url\UrlBuilder') : '';

        $items  = [];
        $catIds = [];

        if ($urlBuilder) {
            $urlBuilder->setStoreId($storeId);
        }

        foreach ($collection as $key => $post) {
            $post = $postRepository->get((int)$post->getId(), (int)$storeId);

            if (!$post) {
                continue;
            }

            $post->setStoreId($storeId);

            $catIds = array_merge($catIds, $post->getCategoryIds());

            $item = new DataObject([
                'id'         => $post->getId(),
                'url'        => $urlBuilder ? $urlBuilder->getPostUrl($post) : $post->getUrl(),
                'title'      => $post->getName(),
                'updated_at' => $post->getUpdatedAt(),
            ]);

            if ($this->alternateConfig->addHreflangToSitemap($storeId)) {
                $item->setAlternates($this->blogStrategy->getBlogAlternates('post', $post));
            }

            $items[] = $item;
        }

        $catIds = array_unique($catIds);

        if (count($catIds)) {
            $catRepository = $this->objectManager->get('Mirasvit\BlogMx\Repository\CategoryRepository');

            $catCollection = $catRepository->getCollection();
            $catCollection->addStoreFilter($storeId)
                ->addVisibilityFilter()
                ->addFieldToFilter('category_id', ['in' => $catIds]); // use only categories with posts

            foreach ($catCollection as $category) {
                $category = $catRepository->get((int)$category->getId(), (int)$storeId);

                if (!$category) {
                    continue;
                }

                $category->setStoreId((int)$storeId);

                $item = new DataObject([
                    'id'         => $category->getId(),
                    'url'        => $urlBuilder ? $urlBuilder->getCategoryUrl($category) : $category->getUrl(),
                    'title'      => $category->getName(),
                    'updated_at' => $category->getUpdatedAt(),
                ]);

                if ($this->alternateConfig->addHreflangToSitemap($storeId)) {
                    $item->setAlternates($this->blogStrategy->getBlogAlternates('category', $category));
                }

                $items[] = $item;
            }
        }

        return $items;
    }
}
