<?php
namespace Windcave\Payments\Block;

use \Magento\Framework\View\Element\Template\Context;

/**
 * Payment information block
 */
class Info extends \Magento\Payment\Block\Info
{
    /**
     * Path to the template file
     *
     * @var string
     */
    protected $_template = 'Windcave_Payments::info/default.phtml';

    /**
     * JSON serializer object
     *
     * @var \Magento\Framework\Serialize\Serializer\Json
     */
    private $_json;

    /**
     * Serializer object
     *
     * @var \Magento\Framework\Serialize\Serializer\Serialize
     */
    private $_serialize;

    /**
     * @param \Magento\Framework\View\Element\Template\Context $context
     * @param array $data
     */
    public function __construct(Context $context, array $data = [])
    {
        parent::__construct($context, $data);
        
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();

        $this->_json = $objectManager->get(\Magento\Framework\Serialize\Serializer\Json::class);
        $this->_serialize = $objectManager->get(\Magento\Framework\Serialize\Serializer\Serialize::class);

        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_logger->info(__METHOD__);
    }

    /**
     * Render as PDF
     *
     * @return string
     */
    public function toPdf()
    {
        $this->setTemplate('Windcave_Payments::info/pdf/default.phtml');
        return $this->toHtml();
    }

    /**
     * Get some specific information in format of array($label => $value)
     *
     * @param bool $isPdf
     * @return array
     */
    public function getSpecificInformation($isPdf = null)
    {
        if ($isPdf) {
            return $this->_prepareSpecificInformationForPdf()->getData();
        }
        return parent::getSpecificInformation();
    }

    /**
     * Prepare information specific to current payment method
     *
     * @param null|\Magento\Framework\DataObject|array $transport
     * @return \Magento\Framework\DataObject
     */
    protected function _prepareSpecificInformation($transport = null)
    {
        $this->_logger->info(__METHOD__);
        if (null !== $this->_paymentSpecificInformation) {
            return $this->_paymentSpecificInformation;
        }

        $data = $this->getInfo()->getAdditionalInformation();
        $decodedData = [];
        foreach ($data as $key => $value) {
            if (strtotime($key)) {
                $decodedValue;
                try {
                    $decodedValue = $this->_json->unserialize($value);
                } catch (\Exception $e) {
                    // TODO: deprecate unserialize completely
                    $decodedValue = $this->_serialize->unserialize($value);
                }
                
                if (!empty($decodedValue)) {
                    $decodedData[$key] = $decodedValue;
                }
            } elseif ($key !== "PxPayHPPUrl") {
                // We don't want to display the URL in the admin panel
                $decodedData[$key] = $value;
            }
        }
        
        $transport = parent::_prepareSpecificInformation($transport);

        unset($decodedData["Currency"]);
        $this->_paymentSpecificInformation = $transport->setData(array_merge($decodedData, $transport->getData()));

        return $this->_paymentSpecificInformation;
    }

    /**
     * Prepare information specific to current payment method for PDF rendering
     *
     * @param null|bool $isPdf
     * @return \Magento\Framework\DataObject
     */
    private function _prepareSpecificInformationForPdf($isPdf = null)
    {
        $this->_logger->info(__METHOD__);
        if (null !== $this->_paymentSpecificInformation) {
            return $this->_paymentSpecificInformation;
        }

        $displayKeys = ["CardName", "CardholderName", "DpsTxnRef"];
        $nameMapping = [
            "CardName" => "Card Type",
            "CardholderName" => "Cardholder Name",
            "DpsTxnRef" => "Transaction Reference",
        ];

        $data = $this->getInfo()->getAdditionalInformation();
        $decodedData = [];
        foreach ($displayKeys as $key) {
            if (isset($data[$key])) {
                $decodedData[$nameMapping[$key]] = $data[$key];
            }
        }
        
        $transport = parent::_prepareSpecificInformation();

        $this->_paymentSpecificInformation = $transport->setData($decodedData);

        return $this->_paymentSpecificInformation;
    }

    /**
     * Returns the PxPayURL if it is stored in the payment info
     *
     * @return string|null
     */
    public function getPxPayUrl()
    {
        $data = $this->getInfo()->getAdditionalInformation();
        if (array_key_exists("PxPayHPPUrl", $data)) {
            return $data["PxPayHPPUrl"];
        }
        return null;
    }
}
