<?php
namespace Windcave\Payments\Helper\Common;

use \Magento\Framework\DataObject;

class PxPost
{
    private const SENSITIVE_FIELDS = [
        "PxPayKey",
        "PostPassword"
    ];

    private const MODULE_NAME = "Windcave_Payments";

    /**
     * @var \Magento\Framework\Module\ModuleListInterface
     */
    private $_moduleList;

    /**
     * @var \Magento\Framework\App\ProductMetadataInterface
     */
    private $_productMetadata;
    
    /**
     * @var \Magento\Framework\HTTP\Client\Curl
     */
    private $_curlClient;

    /**
     * @var \Windcave\Payments\Logger\DpsLogger
     */
    private $_logger;

    /**
     * Constructor
     */
    public function __construct()
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_moduleList = $objectManager->get(\Magento\Framework\Module\ModuleListInterface::class);
        $this->_productMetadata = $objectManager->get(\Magento\Framework\App\ProductMetadataInterface::class);
        $this->_curlClient = $objectManager->get(\Magento\Framework\HTTP\Client\Curl::class);
        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_logger->info(__METHOD__);
    }
    
    /**
     * Returns the module version
     *
     * @return string
     */
    public function getModuleVersion()
    {
        $version = "unknown";
        if ($this->_productMetadata != null) {
            $version = $this->_productMetadata->getVersion();
        }

        if ($this->_moduleList == null) {
            return "M2:" . $version. " ext:unknown";
        }
        return "M2:" . $version . " ext:" . $this->_moduleList->getOne(self::MODULE_NAME)['setup_version'];
    }

    /**
     * Builds and sends request based on supplied request data
     *
     * @param \Magento\Framework\DataObject $requestObject
     * @return string|bool
     */
    public function send($requestObject)
    {
        $this->_logger->info(__METHOD__);
        $requestXml = $this->_buildXml($requestObject);
        $url = $requestObject->getPostUrl();
        return $this->_sendRequest($requestXml, $url);
    }
    
    /**
     * Sends status request
     *
     * @param \Magento\Framework\DataObject $requestObject
     * @return string|bool
     */
    public function sendStatusRequest($requestObject)
    {
        $this->_logger->info(__METHOD__);
        $requestXml = $this->_buildStatusRequestXml($requestObject);
        $url = $requestObject->getPostUrl();
        return $this->_sendRequest($requestXml, $url);
    }
    
    /**
     * Sends the XML request
     *
     * @param \Magento\Framework\DataObject $requestData
     * @param string $postUrl
     * @param int $timeout
     */
    private function _sendRequest($requestData, $postUrl, $timeout = 180)
    {
        $this->_logger->info(__METHOD__ . " postUrl: {$postUrl}");

        $this->_curlClient->setTimeout($timeout);
        $this->_curlClient->addHeader("Content-Type", "application/xml");

        $errorMessage = "";
        try {
            $this->_curlClient->post($postUrl, $requestData);
        } catch (\Exception $ex) {
            $errorMessage = " Error:" . $ex->getMessage();
            $this->_logger->critical(__METHOD__ . $errorMessage);
        }

        $response = $this->_curlClient->getBody();
        $httpcode = $this->_curlClient->getStatus();

        if ($httpcode && substr($httpcode, 0, 2) != "20") {
            $errorMessage = " HTTP CODE: {$httpcode} for URL: {$postUrl}";
            $this->_logger->critical(__METHOD__ . $errorMessage);
        }

        $this->_logger->info(__METHOD__ . " response:" . $response);
        
        return $response;
    }

    /**
     * Adds field into the request object if the value exists
     *
     * @param \Magento\Framework\DataObject $requestObject
     * @param string $fieldName
     * @param string $value
     */
    private function _addFieldIfSet($requestObject, $fieldName, $value)
    {
        if (isset($value)) {
            $requestObject->addChild($fieldName, $value);
        }
    }
    
    /**
     * Builds the XML request from the data
     *
     * @param \Magento\Framework\DataObject $requestData
     */
    private function _buildXml($requestData)
    {
        // <Txn>
        // <PostUsername>Pxpay_HubertFu</PostUsername>
        // <PostPassword>TestPassword</PostPassword>
        // <Amount>1.23</Amount>
        // <InputCurrency>NZD</InputCurrency>
        // <TxnType>Complete</TxnType>
        // <DpsTxnRef>000000600000005b</DpsTxnRef>
        // </Txn>
        $this->_logger->info(__METHOD__);
        
        $requestObject = new \SimpleXMLElement("<Txn></Txn>");
        $requestObject->addChild("PostUsername", $requestData->getUsername());
        $requestObject->addChild("PostPassword", $requestData->getPassword());
        $requestObject->addChild("TxnType", $requestData->getTxnType());
        
        $addNonEmptyValue = function ($name, $value, $maxLength) use (&$requestObject) {
            if (isset($value) && $value) {
                $requestObject->addChild(
                    $name,
                    // phpcs:ignore Magento2.Functions.DiscouragedFunction
                    substr(htmlspecialchars($value, ENT_COMPAT | ENT_XML1), 0, $maxLength)
                );
            }
        };
        
        $addNonEmptyValue("InputCurrency", $requestData->getCurrency(), 4);
        $addNonEmptyValue("Amount", $requestData->getAmount(), 13);
        $addNonEmptyValue("DpsTxnRef", $requestData->getDpsTxnRef(), 16);
        $addNonEmptyValue("DpsBillingId", $requestData->getDpsBillingId(), 16);
        $addNonEmptyValue("TxnId", $requestData->getTxnId(), 16);
        $addNonEmptyValue("MerchantReference", $requestData->getMerchantReference(), 64);
        $addNonEmptyValue("TxnData1", $requestData->getTxnData1(), 255);
        $addNonEmptyValue("TxnData2", $requestData->getTxnData2(), 255);
        $addNonEmptyValue("TxnData3", $requestData->getTxnData3(), 255);
        $addNonEmptyValue("AccountInfo", $requestData->getAccountInfo(), 128);
        $addNonEmptyValue("ClientVersion", $this->getModuleVersion(), 64);

        $requestXml = $requestObject->asXML();
        
        $this->_logger->info(__METHOD__ . " request: {$this->_obscureSensitiveFields($requestObject)}");
        
        return $requestXml;
    }
    
    /**
     * Builds the status request XML
     *
     * @param \Magento\Framework\DataObject $requestData
     */
    private function _buildStatusRequestXml($requestData)
    {
        // <Txn>
        // <PostUsername>Pxpay_HubertFu</PostUsername>
        // <PostPassword>TestPassword</PostPassword>
        // <TxnType>Status</TxnType>
        // <TxnId>000000600000005b</TxnId>
        // </Txn>
        $this->_logger->info(__METHOD__);
    
        $requestObject = new \SimpleXMLElement("<Txn></Txn>");
        $requestObject->addChild("PostUsername", $requestData->getUsername());
        $requestObject->addChild("PostPassword", $requestData->getPassword());
        $requestObject->addChild("TxnType", "Status");
        $requestObject->addChild("TxnId", $requestData->getTxnId());
        $requestObject->addChild("ClientVersion", $this->getModuleVersion());
    
        $requestXml = $requestObject->asXML();
    
        $this->_logger->info(__METHOD__ . " request: {$requestXml}");
    
        return $requestXml;
    }

    /**
     * Obscures the sensitive data
     *
     * @param \SimpleXMLElement $requestObject
     * @return string
     */
    private function _obscureSensitiveFields($requestObject)
    {
        foreach ($requestObject->children() as $child) {
            $name = $child->getName();
            if (in_array($name, self::SENSITIVE_FIELDS)) {
                $child[0] = "****";
            }
        }
        return $requestObject->asXML();
    }
}
