<?php
namespace Windcave\Payments\Helper;

use \Symfony\Component\Console\Helper\ProgressBar;
use \Symfony\Component\Console\Output\OutputInterface;

use \Magento\Framework\Setup\SchemaSetupInterface;

class MigrationHandler
{
    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\Payment\Collection
     */
    private $orderPaymentsCollection = null;

    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\Collection
     */
    private $ordersCollection = null;

    /**
     * @var \Windcave\Payments\Model\ResourceModel\PaymentResult\Collection
     */
    private $paymentResultsCollection = null;

    /**
     * Constructor
     *
     * @param \Magento\Sales\Model\ResourceModel\Order\Payment\Collection $orderPaymentsCollection
     * @param \Magento\Sales\Model\ResourceModel\Order\Collection $ordersCollection
     * @param \Windcave\Payments\Model\ResourceModel\PaymentResult\Collection $paymentResultsCollection
     */
    public function __construct(
        \Magento\Sales\Model\ResourceModel\Order\Payment\Collection $orderPaymentsCollection,
        \Magento\Sales\Model\ResourceModel\Order\Collection $ordersCollection,
        \Windcave\Payments\Model\ResourceModel\PaymentResult\Collection $paymentResultsCollection
    ) {
        $this->orderPaymentsCollection = $orderPaymentsCollection;
        $this->ordersCollection = $ordersCollection;
        $this->paymentResultsCollection = $paymentResultsCollection;
        $this->objectManager = \Magento\Framework\App\ObjectManager::getInstance();
    }

    /**
     * Migrates Payment Methods
     *
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    private function migratePayments(OutputInterface $output)
    {
        $progressBar = new ProgressBar($output->section());
        $oldToNewMethodNames = [
            "paymentexpress_pxpay2" => "windcave_pxpay2",
            "paymentexpress_pxpay2_iframe" => "windcave_pxpay2_iframe",
            "paymentexpress_pxfusion" => "windcave_pxfusion"
        ];

        // Migrating existing order payments
        $payments = $this->orderPaymentsCollection->addFieldToFilter(
            "method",
            [
                "in" => array_keys($oldToNewMethodNames)
            ]
        );
        foreach ($progressBar->iterate($payments) as $payment) {
            $newMethodName = $oldToNewMethodNames[$payment->getMethod()];
            $payment->setMethod($newMethodName);
            $payment->save();
        }
        $progressBar->finish();
    }

    /**
     * Migrates Order Statuses from old table to the new one
     *
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    private function migrateOrderStatuses(OutputInterface $output)
    {
        $progressBar = new ProgressBar($output->section());
        $oldToNewStatuses = [
            "paymentexpress_authorized" => "windcave_authorized",
            "paymentexpress_failed" => "windcave_failed"
        ];

        // Migrating existing orders
        $orders = $this->ordersCollection->addFieldToFilter("status", ["in" => array_keys($oldToNewStatuses)]);
        foreach ($progressBar->iterate($orders) as $order) {
            $newStatus = $oldToNewStatuses[$order->getStatus()];
            $order->setStatus($newStatus);
            $order->save();
        }
        $progressBar->finish();
    }

    /**
     * Migrates Billing Tokens from old table to the new one
     *
     * @param \Magento\Framework\Setup\SchemaSetupInterface $setup
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    private function migrateBillingTokens(SchemaSetupInterface $setup, OutputInterface $output)
    {
        if (!$setup->tableExists('paymentexpress_billingtoken')) {
            return;
        }
        $progressBar = new ProgressBar($output->section(), 100);

        $oldTableName = $setup->getTable('paymentexpress_billingtoken');
        $sqlConnection = $setup->getConnection();
        $select = $sqlConnection->select()->from($oldTableName);
        $columnsToInsert = [
            'entity_id',
            'customer_id',
            'order_id',
            'store_id',
            'masked_card_number',
            'cc_expiry_date',
            'dps_billing_id'
        ];
        $progressBar->advance(5);
        $sqlQuery = $select->insertIgnoreFromSelect($setup->getTable('windcave_billingtoken'), $columnsToInsert, false);
        $sqlConnection->query($sqlQuery);
        $progressBar->finish();
    }

    /**
     * Migrates Payment Results from old table to the new one
     *
     * @param \Magento\Framework\Setup\SchemaSetupInterface $setup
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    private function migratePaymentResults(SchemaSetupInterface $setup, OutputInterface $output)
    {
        if (!$setup->tableExists('paymentexpress_paymentresult')) {
            return;
        }

        $progressBar = new ProgressBar($output->section(), 100);

        $oldTableName = $setup->getTable('paymentexpress_paymentresult');
        $sqlConnection = $setup->getConnection();
        $select = $sqlConnection->select()->from($oldTableName);
        $columnsToInsert = [
            'entity_id',
            'quote_id',
            'reserved_order_id',
            'method',
            'updated_time',
            'dps_transaction_type',
            'dps_txn_ref',
            'raw_xml',
            'user_name',
            'token'
        ];
        $progressBar->advance(5);
        $sqlQuery = $select->insertIgnoreFromSelect(
            $setup->getTable('windcave_paymentresult'),
            $columnsToInsert,
            false
        );
        $sqlConnection->query($sqlQuery);
        $progressBar->finish();
    }

    /**
     * Updates the payment result methods for all the existing orders
     *
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    private function updatePaymentResultMethods(OutputInterface $output)
    {
        $progressBar = new ProgressBar($output->section());
        $oldToNewMethodNames = [
            "paymentexpress_pxpay2" => "windcave_pxpay2",
            "paymentexpress_pxpay2_iframe" => "windcave_pxpay2_iframe",
            "paymentexpress_pxfusion" => "windcave_pxfusion"
        ];

        $payments = $this->paymentResultsCollection->addFieldToFilter(
            "method",
            [
                "in" => array_keys($oldToNewMethodNames)
            ]
        );
        foreach ($progressBar->iterate($payments) as $payment) {
            $newMethodName = $oldToNewMethodNames[$payment->getMethod()];
            $payment->setMethod($newMethodName);
            $payment->save();
        }
        $progressBar->finish();
    }

    /**
     * Runs the migration
     *
     * @param \Magento\Framework\Setup\SchemaSetupInterface $setup
     * @param \Symfony\Component\Console\Output\OutputInterface $output
     */
    public function migrate(SchemaSetupInterface $setup, OutputInterface $output)
    {
        $output->writeln('<info>Migrate payments</info>');
        $this->migratePayments($output);
        $output->writeln('<info>Migrate order statuses</info>');
        $this->migrateOrderStatuses($output);
        $output->writeln('<info>Migrate billing tokens result</info>');
        $this->migrateBillingTokens($setup, $output);
        $output->writeln('<info>Migrate payment results result</info>');
        $this->migratePaymentResults($setup, $output);
        $output->writeln('<info>Migrate payment result methods</info>');
        $this->updatePaymentResultMethods($output);
    }
}
