<?php
namespace Windcave\Payments\Helper\PxPayIFrame;

use \Magento\Framework\App\Helper\AbstractHelper;
use \Magento\Framework\App\Helper\Context;

class Configuration extends AbstractHelper
{
    private const PXPAY2_PATH = "payment/windcave_pxpay2_iframe/";
    private const MODULE_NAME = "Windcave_Payments";

    /**
     * @var \Magento\Framework\Module\ModuleListInterface
     */
    private $_moduleList;

    /**
     * @var \Magento\Framework\App\ProductMetadataInterface
     */
    private $_productMetadata;

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Helper\Context $context
     */
    public function __construct(Context $context)
    {
        parent::__construct($context);
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_moduleList = $objectManager->get(\Magento\Framework\Module\ModuleListInterface::class);
        $this->_productMetadata = $objectManager->get(\Magento\Framework\App\ProductMetadataInterface::class);
        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
    }

    /**
     * Returns the module version
     *
     * @return string
     */
    public function getModuleVersion()
    {
        $version = "unknown";
        if ($this->_productMetadata != null) {
            $version = $this->_productMetadata->getVersion();
        }

        if ($this->_moduleList == null) {
            return "M2:" . $version. " ext:unknown";
        }
        return "M2:" . $version . " ext:" . $this->_moduleList->getOne(self::MODULE_NAME)['setup_version'];
    }

    /**
     * Verifies PxPay configuration
     *
     * @param string|null $storeId
     * @return bool
     */
    public function isValidForPxPay($storeId = null)
    {
        $success = true;
        $len =  !empty($this->getPxPayUserId($storeId)) ? strlen($this->getPxPayUserId($storeId)) : 0;
        if ($len < 1 || $len > 32) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPay Username");
            $success = false;
        }

        $len =  !empty($this->getPxPayKey($storeId)) ? strlen($this->getPxPayKey($storeId)) : 0;
        if ($len < 1 || $len > 64) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPay Key");
            $success = false;
        }

        if (filter_var($this->getPxPayUrl($storeId), FILTER_VALIDATE_URL) === false) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPay URL");
            $success = false;
        }

        return $success;
    }

    /**
     * Verifies PxPost configuration
     *
     * @param string|null $storeId
     * @return bool
     */
    public function isValidForPxPost($storeId = null)
    {
        $success = true;
        $len =  !empty($this->getPxPostUsername($storeId)) ? strlen($this->getPxPostUsername($storeId)) : 0;
        if ($len < 1 || $len > 27) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost Username");
            $success = false;
        }
        $len =  !empty($this->getPxPassword($storeId)) ? strlen($this->getPxPassword($storeId)) : 0;
        if ($len < 1) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost password");
            $success = false;
        }

        if (filter_var($this->getPxPostUrl($storeId), FILTER_VALIDATE_URL) === false) {
            $this->_logger->warn(__METHOD__ . " Invalid PxPost URL");
            $success = false;
        }

        return $success;
    }

    /**
     * Returns folder for lock files
     *
     * @param string|null $storeId
     * @return string
     */
    public function getLocksFolder($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("locksFolder", $storeId);
    }

    /**
     * Returns redirect details for redirect on failure
     *
     * @param string|null $storeId
     * @return array
     */
    public function getRedirectOnErrorDetails($storeId = null)
    {
        $redirectValue = $this->_getPxPay2StoreConfig("redirectonerror", $storeId);
        if ($redirectValue == \Windcave\Payments\Model\Config\Source\RedirectOnErrorOptions::PAYMENT_INFO) {
            return [ 'url' => 'checkout', 'params' => [ '_fragment' => 'payment' ] ];
        }
        return [ 'url' => 'checkout/cart', 'params' => [] ];
    }

    /**
     * Returns where to redirect on payment failure
     *
     * @param string|null $storeId
     * @return int
     */
    public function getRedirectOnErrorMode($storeId = null)
    {
        $redirectValue = $this->_getPxPay2StoreConfig("redirectonerror", $storeId);
        if ($redirectValue == \Windcave\Payments\Model\Config\Source\RedirectOnErrorOptions::PAYMENT_INFO) {
            return 1;
        }
        return 0;
    }

    /**
     * Returns the PxPay Username
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPayUserId($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("pxPayUserId", $storeId);
    }

    /**
     * Returns the PxPay Key
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPayKey($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("pxPayKey", $storeId, true);
    }

    /**
     * Returns the PxPay URL
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPayUrl($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("pxPayUrl", $storeId);
    }

    /**
     * Returns whether PxPay iFrame should be allowed as a payment option
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getEnabled($storeId = null)
    {
        return filter_var($this->_getPxPay2StoreConfig("active", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns whether to allow storing cards for future rebilling
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getAllowRebill($storeId = null)
    {
        return filter_var($this->_getPxPay2StoreConfig("allowRebill", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns the payment type
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPaymentType($storeId = null)
    {
        return (string)$this->_getPxPay2StoreConfig("paymenttype", $storeId);
    }
    
    /**
     * Returns whether for force Account2Account payment option
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getForceA2A($storeId = null)
    {
        return filter_var($this->_getPxPay2StoreConfig("forcea2a", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns the PxPost Username
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostUsername($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("pxpostusername", $storeId);
    }

    /**
     * Returns the PxPost Password
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPassword($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("pxpostpassword", $storeId, true);
    }

    /**
     * Returns the PxPost URL
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPxPostUrl($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("pxposturl", $storeId);
    }

    /**
     * Returns the width of an iFrame for HPP
     *
     * @param string|null $storeId
     * @return int
     */
    public function getIFrameWidth($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("iframeWidth", $storeId);
    }

    /**
     * Returns the height of an iFrame for HPP
     *
     * @param string|null $storeId
     * @return int
     */
    public function getIFrameHeight($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("iframeHeight", $storeId);
    }

    /**
     * Returns the merchant link data
     *
     * @param string|null $storeId
     * @return array
     */
    public function getMerchantLinkData($storeId = null)
    {
        return [
            "Url" => $this->_getPxPay2StoreConfig("merchantLinkUrl", $storeId),
            "Text" => $this->_getPxPay2StoreConfig("merchantLinkText", $storeId)
        ];
    }

    /**
     * Returns the title for "Place Order" button
     *
     * @param string|null $storeId
     * @return string
     */
    public function getPlaceOrderButtonTitle($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("placeOrderButtonTitle", $storeId);
    }

    /**
     * Returns the merchant text
     *
     * @param string|null $storeId
     * @return string
     */
    public function getMerchantText($storeId = null)
    {
        return $this->_getPxPay2StoreConfig("merchantText", $storeId);
    }

    /**
     * Returns whether to force send invoice email to the customer
     *
     * @param string|null $storeId
     * @return bool
     */
    public function getEmailCustomer($storeId = null)
    {
        return filter_var($this->_getPxPay2StoreConfig("emailCustomer", $storeId), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns the logo source
     *
     * @param string $logoPrefix
     * @param string|null $storeId
     * @return string
     */
    public function getLogoSource($logoPrefix, $storeId = null)
    {
        return $this->_getPxPay2StoreConfig($logoPrefix . "Source", $storeId);
    }

    /**
     * Returns the logo alternative text
     *
     * @param string $logoPrefix
     * @param string|null $storeId
     * @return string
     */
    public function getLogoAlt($logoPrefix, $storeId = null)
    {
        return $this->_getPxPay2StoreConfig($logoPrefix . "Alt", $storeId);
    }

    /**
     * Returns the logo height
     *
     * @param string $logoPrefix
     * @param string|null $storeId
     * @return int
     */
    public function getLogoHeight($logoPrefix, $storeId = null)
    {
        return (int)$this->_getPxPay2StoreConfig($logoPrefix . "Height", $storeId);
    }

    /**
     * Returns the logo width
     *
     * @param string $logoPrefix
     * @param string|null $storeId
     * @return int
     */
    public function getLogoWidth($logoPrefix, $storeId = null)
    {
        return (int)$this->_getPxPay2StoreConfig($logoPrefix . "Width", $storeId);
    }

    /**
     * Returns the new auth order status
     *
     * @param string|null $storeId
     * @return string
     */
    public function getNewAuthOrderStatus($storeId = null)
    {
        return (string)$this->_getPxPay2StoreConfig("auth_order_status", $storeId);
    }

    /**
     * Returns PxPay2 store configuration value by name
     *
     * @param string $configName
     * @param string|null $storeId
     * @param bool $isSensitiveData
     * @return string
     */
    private function _getPxPay2StoreConfig($configName, $storeId = null, $isSensitiveData = false)
    {
        $this->_logger->info("Configuration::_getPxPay2StoreConfig storeId argument:" . $storeId);
        
        $value = $this->scopeConfig->getValue(
            self::PXPAY2_PATH . $configName,
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE,
            $storeId
        );
        
        if (!$isSensitiveData) {
            $this->_logger->info(__METHOD__ . " configName:{$configName} storeId:{$storeId} value:{$value}");
        } else {
            $this->_logger->info(__METHOD__ . " configName:{$configName} storeId:{$storeId} value:*****");
        }
        return $value;
    }
}
