<?php
namespace Windcave\Payments\Helper\PxPayIFrame;

use \Magento\Framework\App\Helper\AbstractHelper;
use \Magento\Framework\App\Helper\Context;
use \Magento\Payment\Gateway\Http\Client\Soap;

class UrlCreator
{
    /**
     * @var \Magento\Framework\App\ObjectManager
     */
    private $_objectManager;

    /**
     * @var \Windcave\Payments\Helper\DpsLogger
     */
    private $_logger;

    /**
     * @var \Windcave\Payments\Helper\PxPayIFrame\Configuration
     */
    protected $_configuration;

    /**
     * @var \Windcave\Payments\Helper\PxPayIFrame\Communication
     */
    protected $_communication;

    /**
     * @var \Windcave\Payments\Helper\PaymentUtil
     */
    private $_paymentUtil;

    /**
     * Constructor
     */
    public function __construct()
    {
        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $this->_objectManager = $objectManager;
        $this->_communication = $objectManager->get(\Windcave\Payments\Helper\PxPayIFrame\Communication::class);
        $this->_logger = $objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_configuration = $objectManager->get(\Windcave\Payments\Helper\PxPayIFrame\Configuration::class);
        $this->_paymentUtil = $objectManager->get(\Windcave\Payments\Helper\PaymentUtil::class);
        $this->_logger->info(__METHOD__);
    }

    /**
     * Creates the HPP Url
     *
     * @param \Magento\Quote\Model\Quote $quote
     * @param string $orderId
     * @return string
     */
    public function createUrl(\Magento\Quote\Model\Quote $quote, $orderId)
    {
        $this->_logger->info(__METHOD__);

        $transactionType = $this->_configuration->getPaymentType($quote->getStoreId());
        $forceA2A = $this->_configuration->getForceA2A($quote->getStoreId());
        $requestData = $this->_buildPxPayRequestData($quote, $transactionType, $forceA2A, $orderId);

        $responseXml = $this->_communication->getPxPay2Page($requestData);

        $responseXmlElement = simplexml_load_string($responseXml);
        if (!$responseXmlElement) {
            $error = "Invalid response from Windcave: " . $responseXml;
            $this->_logger->critical(__METHOD__ . " " . $error);
            return "";
        }

        if ($responseXmlElement['valid'] != "1" || !$responseXmlElement->URI) {
            $error = "Failed to get the Payment Url";
            // <Request valid="1"><Reco>W2</Reco>
            // <ResponseText>No Account2Account Account Setup For Payment Currency</ResponseText></Request>
            if (isset($responseXmlElement->Reco) || isset($responseXmlElement->ResponseText)) {
                $error = "Error from Windcave: ReCo: " . $responseXmlElement->Reco . " ResponseText:" .
                     $responseXmlElement->ResponseText;
            } elseif (isset($responseXmlElement->URI)) {
                $error = "Error from Windcave: " . $responseXmlElement->URI;
            }

            $this->_logger->critical(__METHOD__ . " " . $error);

            return "";
        }

        return (string)$responseXmlElement->URI;
    }

    /**
     * Gets and validates the boolean value from the array
     *
     * @param array $array
     * @param string $fieldName
     *
     * @return bool
     */
    private function _getBoolValue($array, $fieldName)
    {
        if (!isset($array)) {
            return false;
        }
        if (!isset($array[$fieldName])) {
            return false;
        }

        return filter_var($array[$fieldName], FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Gets and validates the string value from the array
     *
     * @param array $array
     * @param string $fieldName
     *
     * @return string
     */
    private function _getStringValue($array, $fieldName)
    {
        if (!isset($array)) {
            return "";
        }
        if (!isset($array[$fieldName])) {
            return "";
        }

        return filter_var(
            $array[$fieldName],
            FILTER_UNSAFE_RAW,
            FILTER_FLAG_STRIP_LOW | FILTER_FLAG_STRIP_HIGH
        );
    }

    /**
     * Builds PxPay request data
     *
     * @param \Magento\Quote\Model\Quote $quote
     * @param string $transactionType
     * @param bool $forceA2A
     * @param string $orderId
     * @return \Magento\Framework\DataObject
     */
    private function _buildPxPayRequestData(\Magento\Quote\Model\Quote $quote, $transactionType, $forceA2A, $orderId)
    {
        $orderIncrementId = $orderId ?: $quote->getReservedOrderId();
        $this->_logger->info(
            __METHOD__ . " orderIncrementId:{$orderIncrementId} " .
            "transactionType:{$transactionType} forceA2A:{$forceA2A}"
        );

        $currency = $quote->getQuoteCurrencyCode();
        $amount = $this->_paymentUtil->formatCurrency($quote->getGrandTotal(), $currency);

        $payment = $quote->getPayment();
        $additionalInfo = $payment->getAdditionalInformation();
        if (!isset($additionalInfo)) {
            $additionalInfo = [];
        }

        $dpsBillingId = "";
        $useSavedCard = $this->_getBoolValue($additionalInfo, "UseSavedCard");
        if ($useSavedCard) {
            $dpsBillingId = $this->_getStringValue($additionalInfo, "DpsBillingId");
        }
        $enableAddBillCard = $this->_getBoolValue($additionalInfo, "EnableAddBillCard");

        $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
        $dataBag = $objectManager->create(\Magento\Framework\DataObject::class);
        $dataBag->setForceA2A(false);
        if ($transactionType == "Purchase" && $forceA2A) {
            $dataBag->setForceA2A(true);
        }

        $txnId = substr(uniqid(rand()), 0, 16);
        $dataBag->setTxnId($txnId); // quote cannot be used as txnId. As quote may pay failed.

        // <TxnId>ABC123</TxnId>
        // <TxnData1>John Doe</TxnData1>
        // <TxnData2>0211111111</TxnData2>
        // <TxnData3>98 Anzac Ave, Auckland 1010</TxnData3>

        $dataBag->setAmount($amount);
        $dataBag->setCurrency($currency);
        $dataBag->setTransactionType($transactionType);
        $dataBag->setOrderIncrementId($orderIncrementId);
        $dataBag->setQuoteId($quote->getId());
        $dataBag->setDpsBillingId($dpsBillingId);
        $dataBag->setEnableAddBillCard($enableAddBillCard);

        $customerInfo = $this->_loadCustomerInfo($quote);
        $dataBag->setCustomerInfo($customerInfo);
        $this->_logger->info(__METHOD__ . " dataBag:" . var_export($dataBag, true));
        return $dataBag;
    }

    /**
     * Loads the customer information
     *
     * @param \Magento\Quote\Model\Quote $quote
     * @return \Magento\Framework\DataObject
     */
    private function _loadCustomerInfo(\Magento\Quote\Model\Quote $quote)
    {
        $customerId = $quote->getCustomerId();
        $this->_logger->info(__METHOD__ . " customerId:{$customerId}");
        $customerInfo = $this->_objectManager->create(\Magento\Framework\DataObject::class);

        $customerInfo->setId($customerId);

        $customerInfo->setName($this->_getCustomerName($quote));
        try {
            $address = $quote->getBillingAddress();
            if ($address) {
                $customerInfo->setPhoneNumber($address->getTelephone());
                $customerInfo->setEmail($address->getEmail());
                $streetFull = $this->_implode(" ", $address->getStreet()) . " " . $address->getCity() . ", " .
                     $address->getRegion() . " " . $address->getPostcode() . " " . $address->getCountryId();

                $customerInfo->setAddress($streetFull);
            }
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            $this->_logger->critical($e->_toString());
        }

        return $customerInfo;
    }

    /**
     * Retrieve customer name
     *
     * @param \Magento\Quote\Model\Quote $quote
     * @return string
     */
    private function _getCustomerName(\Magento\Quote\Model\Quote $quote)
    {
        if ($quote->getBillingAddress()->getName()) {
            $customerName = $quote->getBillingAddress()->getName();
        } elseif ($quote->getShippingAddress()->getName()) {
            $customerName = $quote->getShippingAddress()->getName();
        } elseif ($quote->getCustomerFirstname()) {
            $customerName = $quote->getCustomerFirstname() . ' ' . $quote->getCustomerLastname();
        } else {
            $customerName = (string)__('Guest');
        }

        $this->_logger->info(__METHOD__ . " customerName:{$customerName}");
        return $customerName;
    }

    /**
     * Implodes the array
     *
     * @param string $token
     * @param array $dta_array
     */
    private function _implode($token, $dta_array)
    {
        if (version_compare(PHP_VERSION, '7.4.3') >= 0) {
            return implode($token, $dta_array);
        } else {
            return implode($dta_array, $token);
        }
    }
}
