<?php
namespace Windcave\Payments\Model;

use \Magento\Checkout\Model\ConfigProviderInterface;

// Invoked by Magento\Checkout\Block\Onepage::getCheckoutConfig
class PxPay2ConfigProvider implements ConfigProviderInterface
{

    /**
     * @var \Windcave\Payments\Logger\DpsLogger
     */
    private $_logger;

    /**
     * @var \Magento\Framework\App\ObjectManager
     */
    private $_objectManager;

    /**
     * @var \Windcave\Payments\Helper\Configuration
     */
    private $_configuration;

    /**
     * @var \Windcave\Payments\Block\MerchantLogo
     */
    private $_merchantLogo;

    /**
     * Constructor
     *
     * @param \Magento\Framework\ObjectManagerInterface $objectManager
     */
    public function __construct(\Magento\Framework\ObjectManagerInterface $objectManager)
    {
        $this->_objectManager = $objectManager;
        $this->_configuration = $this->_objectManager->get(\Windcave\Payments\Helper\Configuration::class);
        $this->_logger = $this->_objectManager->get(\Windcave\Payments\Logger\DpsLogger::class);
        $this->_merchantLogo = $this->_objectManager->get(\Windcave\Payments\Block\MerchantLogo::class);
        $this->_logger->info(__METHOD__);
    }

    /**
     * Returns the configuration
     *
     * @return array
     */
    public function getConfig()
    {
        $this->_logger->info(__METHOD__);
        $session = $this->_objectManager->get(\Magento\Checkout\Model\Session::class);
        $quote = $session->getQuote();
        $quoteId = $quote->getId();
        $customerId = $quote->getCustomerId();
        
        $customerSession = $this->_objectManager->get(\Magento\Customer\Model\Session::class);
        $isRebillEnabled = ($customerSession->isLoggedIn() && $this->_configuration->getAllowRebill());
        // not show card configuration when rebill is false or A2A is disabled.
        $showCardOptions = $isRebillEnabled && !$this->_configuration->getForceA2A();
        
        $paymentUtil = $this->_objectManager->get(\Windcave\Payments\Helper\PaymentUtil::class);
        
        $logos = [];
        
        for ($i = 1; $i <= 5; $i++) {
            $this->_merchantLogo->setLogoPathPrefix("merchantLogo{$i}");
            $url = $this->_merchantLogo->getLogoUrl();
            if (empty($url)) {
                continue;
            }
            $logos[] = [
                "Url" => $url,
                "Alt" => $this->_merchantLogo->getLogoAlt(),
                "Width" => $this->_merchantLogo->getLogoWidth(),
                "Height" => $this->_merchantLogo->getLogoHeight()
            ];
        }
        
        $merchantUICustomOptions = [
            'linkData' => $this->_configuration->getMerchantLinkData(),
            'logos' => $logos,
            'text' => $this->_configuration->getMerchantText()
        ];
        
        return [
            'payment' => [
                'windcave' => [
                    'redirectUrl' => $paymentUtil->buildRedirectUrl($quoteId),
                    'savedCards' => $this->_loadSavedCards($customerId),
                    'isRebillEnabled' => $isRebillEnabled,
                    'showCardOptions' => $showCardOptions,
                    'placeOrderButtonTitle' => $this->_configuration->getPlaceOrderButtonTitle(),
                    'merchantUICustomOptions' => $merchantUICustomOptions,
                    'method' => \Windcave\Payments\Model\Payment::PXPAY_CODE
                ]
            ]
        ];
    }

    /**
     * Loads the customer's saved cards
     *
     * @param string $customerId
     * @return array
     */
    private function _loadSavedCards($customerId)
    {
        $this->_logger->info(__METHOD__ . " customerId:{$customerId}");
        $savedCards = [];
        
        if (!empty($customerId)) { // do not access database if the order is processed by guest, to improve performance.
            $billingModel = $this->_objectManager->create(\Windcave\Payments\Model\BillingToken::class);
            $billingModelCollection = $billingModel->getCollection()->addFieldToFilter('customer_id', $customerId);
            $billingModelCollection->getSelect()->group(
                [
                'masked_card_number',
                'cc_expiry_date'
                ]
            );
            
            foreach ($billingModelCollection as $item) {
                $maskedCardNumber = trim($item->getMaskedCardNumber());
                $ccExpiryDate = trim($item->getCcExpiryDate());
                if (!empty($maskedCardNumber)) {
                    $savedCards[] = [
                        "billing_token" => $item->getDpsBillingId(),
                        "card_number" => $maskedCardNumber,
                        "expiry_date" => $ccExpiryDate,
                        "card_info" => $maskedCardNumber . " Expiry Date:" . $ccExpiryDate
                    ];
                }
            }
        }
        
        return $savedCards;
    }
}
