<?php

namespace Asics\Verifone\Controller\Standard;

use Asics\Verifone\Helper\Configuration;
use Exception;
use Magento\Checkout\Model\Type\Onepage;
use Magento\Customer\Model\Session;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Controller\Result\JsonFactory;
use Magento\Framework\Controller\ResultInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Quote\Api\CartManagementInterface;
use Magento\Quote\Api\CartRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\OrderFactory;
use Psr\Log\LoggerInterface;
use Verifone\Hosted\Controller\Standard\Response;
use Verifone\Hosted\Model\Verifone;
use Verifone\Hosted\Helper\Verifone as VerifoneHelper;
use Magento\Checkout\Model\Session as CheckoutModelSession;

class ResponseRewrite extends Response
{
    const STATUS_PENDING = 'pending';

    /**
     * @var Configuration
     */
    protected $_configuration;

    /**
     * ResponseRewrite constructor.
     *
     * @param Context $context
     * @param Session $customerSession
     * @param CheckoutModelSession $checkoutSession
     * @param CartRepositoryInterface $quoteRepository
     * @param OrderFactory $orderFactory
     * @param LoggerInterface $logger
     * @param Verifone $paymentMethod
     * @param VerifoneHelper $checkoutHelper
     * @param CartManagementInterface $cartManagement
     * @param JsonFactory $resultJsonFactory
     * @param Configuration $configuration
     */
    public function __construct(
        Context                 $context,
        Session                 $customerSession,
        CheckoutModelSession    $checkoutSession,
        CartRepositoryInterface $quoteRepository,
        OrderFactory            $orderFactory,
        LoggerInterface         $logger,
        Verifone                $paymentMethod,
        VerifoneHelper          $checkoutHelper,
        CartManagementInterface $cartManagement,
        JsonFactory             $resultJsonFactory,
        Configuration           $configuration
    ) {
        $this->_configuration = $configuration;
        parent::__construct(
            $context,
            $customerSession,
            $checkoutSession,
            $quoteRepository,
            $orderFactory,
            $logger,
            $paymentMethod,
            $checkoutHelper,
            $cartManagement,
            $resultJsonFactory
        );
    }

    /**
     * @return ResponseInterface|ResultInterface|void
     * @throws LocalizedException
     * @throws NoSuchEntityException
     * @throws Exception
     */
    public function execute()
    {
        // Get params from response
        $params = $this->getRequest()->getParams();
        $paymentMethod = $this->getPaymentMethod();
        $result = $this->getCheckoutHelper()->checkPaymentResponse($paymentMethod, $params);

        if (!$result || $result['status'] !== "COMPLETED") {
            $objectManager = ObjectManager::getInstance();
            $messageManager = $objectManager->get('Magento\Framework\Message\ManagerInterface');
            $messageManager->addErrorMessage('Something went wrong. Your payment was not successful. Please try again!');
            $returnUrl = $this->getCheckoutHelper()->getUrl('checkout/cart');

            if ($this->getPaymentSettings('active')
                && $this->getPaymentSettings('account_settings/order_before_redirect')
            ) {
                $lastRealOrder = $this->_checkoutSession->getLastRealOrder();
                if ($lastRealOrder) {
                    $payment = $lastRealOrder->getPayment();
                    if (!isset($payment) || $payment === null || $payment->getMethod() !== Verifone::CODE) {
                        return $this->getResponse()->setRedirect($returnUrl);
                    }
                    $info = $payment->getAdditionalInformation();
                    if (isset($info) && !empty($info['verifone_order_number'])) {
                        return $this->getResponse()->setRedirect($returnUrl);
                    }
                    if ($lastRealOrder->getState() === Order::STATE_NEW
                        && $lastRealOrder->getStatus() === self::STATUS_PENDING) {
                        $this->_cancelPayment();
                        $returnUrl = $this->getCheckoutHelper()->getUrl('checkout', ['_fragment' => 'payment']);
                        return $this->getResponse()->setRedirect($returnUrl);
                    }
                }
            }
            return $this->getResponse()->setRedirect($returnUrl);
        }

        $transactionId = $result['transaction_id'];
        $quoteId = $this->getQuote()->getId()??$result['merchant_reference'];
        $quote = $quoteId ? $this->_quoteRepository->get($quoteId) : $this->getCheckoutHelper()->getQuoteByIncrementId($transactionId);

        if ($quote) {
            $this->getCheckoutSession()->replaceQuote($quote);
        }

        // Get payment method code
        $code = $paymentMethod->getCode();
        $order = $this->getCheckoutHelper()->getOrderByIncrementId($quote->getReservedOrderId());

        if (!$order && !$quote) {
            $this->messageManager->addExceptionMessage(
                new Exception(__('Missing order and quote data!')),
                __('Missing order and quote data!')
            );
        }

        if ($quote && !$order) {
            if ($this->getCustomerSession()->isLoggedIn()) {
                if (isset($params['email']) && !empty($params['email'])) {
                    $quote->setCustomerEmail($params['email']);
                }
                $quote->setCheckoutMethod(Onepage::METHOD_CUSTOMER);
            } else {
                $quote->setCheckoutMethod(Onepage::METHOD_GUEST);
            }

            $quote->setPaymentMethod($code);
            $quote->getPayment()->importData(['method' => $code]);
            $this->_quoteRepository->save($quote);
        }

        if (!$order) {
            try {
                //Quote has been updated. From now, we process the checkout
                $this->initCheckout();
                $this->_cartManagement->placeOrder(
                    $quote->getId(),
                    $quote->getPayment()
                );
                $order = $this->getOrder();

                if (!$this->_checkoutHelper->checkVerifoneTransactionIdInOrderLogHistory($transactionId, $order)) {
                    $order->addStatusHistoryComment(__('Order created when redirected from payment page. Transaction ID: "%1"', $transactionId));
                }

            } catch (Exception $e) {
            }
        } else {
            // set the checkoutSession for the redirect
            $this->getCheckoutSession()
                 ->setLastSuccessQuoteId($quote->getId())
                 ->setLastQuoteId($quote->getId())
                 ->setLastRealOrderId($order->getIncrementId())
                 ->setLastOrderId($order->getId())
                 ->setLastOrderStatus($order->getStatus());
        }

        if ($order && !$this->_checkoutHelper->checkVerifoneTransactionIdInOrderLogHistory($transactionId, $order)) {
            $order->setExtOrderId($transactionId);
            $order->save();

            $manualCapture = $this->getCheckoutHelper()->checkManualCapture();

            $paymentMethod->postProcessing($order, $result, $manualCapture, $result['payment_method_used'], (int) $result['amount']);
        }

        $returnUrl = $this->getCheckoutHelper()->getUrl('checkout/onepage/success');

        $this->getResponse()->setRedirect($returnUrl);
    }

    /**
     * @param $path
     * @return mixed
     */
    public function getPaymentSettings($path)
    {
        return $this->_configuration->_getVerifoneStoreConfig($path);
    }
}
