<?php

declare(strict_types=1);

namespace BrittainWynyard\CatalogSuperStyle\Plugin\ProductList;

use Magento\Catalog\Block\Product\ProductList\Related;
use Magento\Catalog\Block\Product\ProductList\Upsell;
use Magento\Catalog\Block\Product\ProductList\Crosssell;
use Magento\CatalogWidget\Block\Product\ProductsList;
use Magento\Catalog\Block\Product\ListProduct;
use BrittainWynyard\CatalogSuperStyle\Helper\Data;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Psr\Log\LoggerInterface;

class SuperSwatchLoader
{
    /**
     * @var Data
     */
    private $dataHelper;

    /**
     * @var array
     */
    private $processedBlocks = [];

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * Constructor
     *
     * @param Data $dataHelper
     * @param LoggerInterface $logger
     */
    public function __construct(
        Data $dataHelper,
        LoggerInterface $logger
    ) {
        $this->dataHelper = $dataHelper;
        $this->logger = $logger;
    }

    /**
     * Plugin for Related products getItems method
     *
     * @param Related $subject
     * @param mixed $result
     * @return mixed
     */
    public function afterGetItems(
        Related $subject,
        $result
    ) {
        $this->processProductCollection($result, get_class($subject));
        return $result;
    }

    /**
     * Plugin for Product list getLoadedProductCollection method
     *
     * @param ListProduct $subject
     * @param mixed $result
     * @return mixed
     */
    public function afterGetLoadedProductCollection(
        ListProduct $subject,
        $result
    ) {
        $this->processProductCollection($result, get_class($subject));
        return $result;
    }

    /**
     * Plugin for Widget products getProductCollection method
     *
     * @param ProductsList $subject
     * @param mixed $result
     * @return mixed
     */
    public function afterGetProductCollection(
        ProductsList $subject,
        $result
    ) {
        $this->processProductCollection($result, get_class($subject));
        return $result;
    }

    /**
     * Plugin for Widget products createCollection method
     *
     * @param ProductsList $subject
     * @param mixed $result
     * @return mixed
     */
    public function afterCreateCollection(
        ProductsList $subject,
        $result
    ) {
        $this->processProductCollection($result, get_class($subject));
        return $result;
    }

    /**
     * Plugin for getAllItems method (used by rule-based blocks)
     *
     * @param object $subject
     * @param mixed $result
     * @return mixed
     */
    public function afterGetAllItems(
        $subject,
        $result
    ) {
        // Process array result directly
        if (is_array($result) && !empty($result)) {
            // Process the array items directly without ArrayObject wrapper
            $this->processProductCollection($result, get_class($subject));
        }
        
        return $result;
    }

    /**
     * Plugin for getItemCollection method (used by upsell/crosssell blocks)
     *
     * @param object $subject
     * @param mixed $result
     * @return mixed
     */
    public function afterGetItemCollection(
        $subject,
        $result
    ) {
        $this->processProductCollection($result, get_class($subject));
        return $result;
    }

    /**
     * Process product collection to batch-load super swatch counts
     *
     * @param mixed $collection
     * @param string $blockClass
     * @return void
     */
    private function processProductCollection($collection, string $blockClass): void
    {
        if (!$collection) {
            return;
        }

        // Handle different collection types
        $items = [];
        if (is_array($collection)) {
            $items = $collection;
        } elseif (method_exists($collection, 'getItems')) {
            $items = $collection->getItems();
        } elseif ($collection instanceof \Traversable) {
            $items = iterator_to_array($collection);
        } else {
            return;
        }

        // Create unique identifier for this collection to avoid processing it multiple times
        if (is_object($collection)) {
            $collectionId = spl_object_hash($collection);
        } else {
            // For arrays, create a simple hash based on block class and item count/types
            // Avoid serialize() as collections may contain non-serializable objects like closures
            $hashData = [
                'block_class' => $blockClass,
                'collection_type' => is_array($collection) ? 'array' : 'object',
                'item_count' => is_array($collection) ? count($collection) : 0
            ];
            
            // Add some basic info about first few items if available
            if (is_array($collection) && !empty($collection)) {
                $sampleItems = array_slice($collection, 0, 3);
                foreach ($sampleItems as $idx => $item) {
                    if (is_object($item)) {
                        /** @var \Magento\Catalog\Model\Product $item */
                        $hashData['item_' . $idx] = $item->getId();
                    }
                }
            }
            
            $collectionId = hash('sha256', json_encode($hashData));
        }
        
        if (isset($this->processedBlocks[$collectionId])) {
            return; // Already processed
        }

        $this->processedBlocks[$collectionId] = true;

        $products = [];
        $configurableProducts = [];
        $superStyleIds = [];

        // Extract products from the items and filter configurable ones
        foreach ($items as $product) {
            $products[] = $product;
            if ($product->getTypeId() === Configurable::TYPE_CODE && $product->getData('super_style_id')) {
                $configurableProducts[] = $product;
                $superStyleIds[] = $product->getData('super_style_id');
            }
        }

        $totalProducts = count($products);
        $configurableCount = count($configurableProducts);
        $uniqueSuperStyleIds = array_unique($superStyleIds);

        // Only batch load if we have configurable products with super_style_id
        if (!empty($configurableProducts)) {

            // Batch load the super swatch counts
            $this->dataHelper->loadSwatchCounts($configurableProducts);

            $processedCount = 0;
            $cachedCounts = [];

            // Set the cached counts on each configurable product
            foreach ($configurableProducts as $product) {
                $superStyleId = $product->getData('super_style_id');
                if ($superStyleId) {
                    $count = $this->dataHelper->getSuperSwatchCount($superStyleId);
                    $product->setData('super_swatch_related_count', $count);
                    
                    // Also set a flag to indicate the data is loaded
                    $product->setData('super_swatch_count_loaded', true);
                    
                    $cachedCounts[$superStyleId] = $count;
                    $processedCount++;
                }
            }

        }
    }
}
