<?php
/**
 * Copyright © Overdose Digital. All rights reserved.
 */
namespace Overdose\Klaviyo\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Klaviyo\Reclaim\Helper\Data as KlaviyoHelper;
use Klaviyo\Reclaim\Helper\ScopeSetting;
use Klaviyo\Reclaim\Helper\Logger as KlaviyoLogger;
use Magento\Framework\Session\SessionManagerInterface;
use Magento\Store\Model\StoreManagerInterface;
use Psr\Log\LoggerInterface;
use Magento\Framework\Stdlib\CookieManagerInterface;
use Magento\Framework\Stdlib\Cookie\CookieMetadataFactory;

class Data extends AbstractHelper
{
    const EVENT_LOGGED_IN = 'Logged In';
    const KLAVIYO_API_EVENTS_URL = 'https://a.klaviyo.com/api/events';
    /**
     * @var KlaviyoHelper
     */
    protected $klaviyoHelper;

    /**
     * @var ScopeSetting
     */
    protected $scopeSetting;

    /**
     * @var SessionManagerInterface
     */
    protected $coreSession;

    /**
     * @var StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var KlaviyoLogger
     */
    protected $logger;

    protected $cookieManager;
    protected $cookieMetadataFactory;

    /**
     * @param Context $context
     * @param KlaviyoHelper $klaviyoHelper
     * @param ScopeSetting $scopeSetting
     * @param SessionManagerInterface $coreSession
     * @param StoreManagerInterface $storeManager
     * @param KlaviyoLogger $logger
     */
    public function __construct(
        Context $context,
        KlaviyoHelper $klaviyoHelper,
        ScopeSetting $scopeSetting,
        SessionManagerInterface $coreSession,
        StoreManagerInterface $storeManager,
        KlaviyoLogger $logger,
        CookieManagerInterface $cookieManager,
        CookieMetadataFactory  $cookieMetadataFactory
    ) {
        $this->klaviyoHelper = $klaviyoHelper;
        $this->scopeSetting = $scopeSetting;
        $this->coreSession = $coreSession;
        $this->storeManager = $storeManager;
        $this->logger = $logger;
        $this->cookieManager = $cookieManager;
        $this->cookieMetadataFactory = $cookieMetadataFactory;
        parent::__construct($context);
    }

    /**
     * Track customer login event in Klaviyo
     *
     * @param \Magento\Customer\Model\Customer $customer
     * @return bool
     */
    public function trackLoginEvent($customer)
    {
        try {
            $storeId = $this->storeManager->getStore()->getId();
            $referrer = $this->cookieManager->getCookie('redirect_link');
            if (!$referrer) {
                return false;
            }
            $metadata = $this->cookieMetadataFactory->createPublicCookieMetadata()
                                                    ->setPath('/');
            $this->cookieManager->deleteCookie('redirect_link', $metadata);

            $customerProperties = [
                '$email' => $customer->getEmail(),
                '$first_name' => $customer->getFirstname(),
                '$last_name' => $customer->getLastname(),
                '$id' => $customer->getId()
            ];

            $eventProperties = [
                'referrer' => $referrer,
                'store_id' => $storeId,
                'store_name' => $this->storeManager->getStore()->getName()
            ];

            $result = $this->sendEventToKlaviyo(
                self::EVENT_LOGGED_IN,
                $customerProperties,
                $eventProperties,
                null,
                $storeId
            );

            return $result;
        } catch (\Exception $e) {
            $this->logger->log('Error tracking login event in Klaviyo: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Send event to Klaviyo using the events API
     *
     * @param string $event
     * @param array $customerProperties
     * @param array $properties
     * @param int|null $timestamp
     * @param int|null $storeId
     * @return bool
     * @throws \Exception
     */
    protected function sendEventToKlaviyo($event, $customerProperties = [], $properties = [], $timestamp = null, $storeId = null)
    {
        if ((!array_key_exists('$email', $customerProperties) || empty($customerProperties['$email']))
            && (!array_key_exists('$id', $customerProperties) || empty($customerProperties['$id']))
            && (!array_key_exists('$exchange_id', $customerProperties) || empty($customerProperties['$exchange_id'])))
        {
            throw new \Exception('You must identify a user by email or ID.');
        }

        $data = [
            'data' => [
                'type'       => 'event',
                'attributes' => [
                    'properties' => $properties,
                    'metric'     => [
                        'data' => [
                            'type'       => 'metric',
                            'attributes' => [
                                'name' => $event
                            ]
                        ]
                    ],
                    'profile'    => [
                        'data' => [
                            'type'       => 'profile',
                            'attributes' => [
                                'email'      => $customerProperties['$email'],
                                'first_name' => $customerProperties['$first_name'],
                                'last_name'  => $customerProperties['$last_name']
                            ]
                        ]
                    ],
                    'time'       => !is_null($timestamp) ? $timestamp : date('c'),
                    'value'      => 0,
                    'unique_id'  => uniqid('event_' . strtolower(str_replace(' ', '', $event)), true)
                ]
            ]
        ];


        $url = self::KLAVIYO_API_EVENTS_URL;

        // Log the data being sent to Klaviyo
        $this->logger->log('Sending event to Klaviyo: ' . json_encode([
            'event' => $event,
            'data' => $data
        ]));

        $curl = curl_init();
        $encodedData = json_encode($data);

        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => $encodedData,
            CURLOPT_USERAGENT => 'Klaviyo/1.0',
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/vnd.api+json',
                'Authorization: Klaviyo-API-Key ' . $this->scopeSetting->getPrivateApiKey($storeId),
                'Accept: application/vnd.api+json',
                'Revision: 2025-04-15'
            ],
        ]);

        // Submit the request
        $response = curl_exec($curl);
        $statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $err = curl_errno($curl);

        // Log the response from Klaviyo
        $this->logger->log('Response from Klaviyo: ' . json_encode([
            'statusCode' => $statusCode,
            'response' => $response
        ]));

        if ($err) {
            throw new \Exception(curl_error($curl));
        }

        // Close cURL session handle
        curl_close($curl);

        if ($statusCode >= 200 && $statusCode < 300) {
            return true;
        } else {
            $this->logger->log('Error sending event to Klaviyo: ' . $response);
            return false;
        }
    }
}
