<?php

namespace Overdose\Quote\Plugin\Controller\Cart;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Checkout\Model\Cart as CustomerCart;
use Magento\Checkout\Model\Cart\RequestQuantityProcessor;
use Magento\Checkout\Model\Session;
use Magento\Framework\App\Action\Context;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResponseInterface;
use Magento\Framework\Controller\ResultInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\Data\Form\FormKey\Validator;
use Magento\Framework\App\ResourceConnection;
use Magento\CatalogInventory\Model\StockRegistry;

class Add extends \Magento\Checkout\Controller\Cart\Add
{
    /**
     * @var ResourceConnection
     */
    protected ResourceConnection $resource;

    /**
     * @var StockRegistry
     */
    protected StockRegistry $stockRegistry;

    /**
     * @param Context $context
     * @param ScopeConfigInterface $scopeConfig
     * @param Session $checkoutSession
     * @param StoreManagerInterface $storeManager
     * @param Validator $formKeyValidator
     * @param CustomerCart $cart
     * @param ProductRepositoryInterface $productRepository
     * @param ResourceConnection $resource
     * @param StockRegistry $stockRegistry
     * @param RequestQuantityProcessor|null $quantityProcessor
     */
    public function __construct(
        Context                    $context,
        ScopeConfigInterface       $scopeConfig,
        Session                    $checkoutSession,
        StoreManagerInterface      $storeManager,
        Validator                  $formKeyValidator,
        CustomerCart               $cart,
        ProductRepositoryInterface $productRepository,
        ResourceConnection         $resource,
        StockRegistry              $stockRegistry,
        ?RequestQuantityProcessor  $quantityProcessor = null
    ) {
        $this->stockRegistry = $stockRegistry;
        $this->resource = $resource;
        parent::__construct
        (
            $context,
            $scopeConfig,
            $checkoutSession,
            $storeManager,
            $formKeyValidator,
            $cart,
            $productRepository,
            $quantityProcessor
        );
    }

    public function execute()
    {
        $quoteId = $this->_checkoutSession->getQuote()->getId();
        $productId = $this->getRequest()->getParam('product');
        $addToCartQty = (int)$this->getRequest()->getParam('qty');
        $allowMaxQty = (int)$this->getRequest()->getParam('od-qty-max-allow');
        if (!$allowMaxQty) {
            $stockData = $this->stockRegistry->getStockItem($productId, $this->_storeManager->getWebsite()->getId());
            $allowMaxQty = (int)$stockData->getMaxSaleQty();
        }
        if ($quoteId && $productId && $addToCartQty && $allowMaxQty) {
            $select = $this->getConnection()->select()->from(
                ['qi' => $this->getConnection()->getTableName('quote_item')],
                ['product_id', 'qty']
            )->where(
                'qi.product_id = :product_id and qi.quote_id = :quote_id'
            );
            $bind = [
                'product_id' => (int)$productId,
                'quote_id' => (int)$quoteId,
            ];
            $result = $this->getConnection()->fetchAll($select, $bind);
            $currentQty = 0;
            foreach ($result as $quoteItem) {
                $currentQty += (int)$quoteItem['qty'];
            }

            $qtyCheck = $currentQty + $addToCartQty;
            if ($qtyCheck > $allowMaxQty) {
                $url = $this->_checkoutSession->getRedirectUrl(true);
                if (!$url) {
                    $url = $this->_redirect->getRedirectUrl($this->getCartUrl());
                }
                return $this->goBackWithMessage($url, $quoteId, $qtyCheck, $allowMaxQty);
            }
        }
        return parent::execute();
    }

    /**
     * @return \Magento\Framework\DB\Adapter\AdapterInterface
     */
    public function getConnection()
    {
        return $this->resource->getConnection(\Magento\Framework\App\ResourceConnection::DEFAULT_CONNECTION);
    }

    /**
     * @param $backUrl
     * @param $quoteId
     * @param $qtyCheck
     * @param $allowMaxQty
     * @return ResponseInterface|\Magento\Framework\Controller\Result\Redirect
     */
    protected function goBackWithMessage($backUrl, $quoteId, $qtyCheck, $allowMaxQty)
    {
        if (!$this->getRequest()->isAjax()) {
            return parent::_goBack($backUrl);
        }
        $result = [];
        $result['odMaxQtyError'] = 'The requested quantity exceeds the maximum quantity allowed in shopping cart';
        $result['quote_id'] = $quoteId;
        $result['check_qty'] = $qtyCheck;
        $result['allow_max_qty'] = $allowMaxQty;

        $this->getResponse()->representJson(
            json_encode($result)
        );

        return $this->getResponse();
    }

    /**
     * Returns cart url
     *
     * @return string
     */
    private function getCartUrl()
    {
        return $this->_url->getUrl('checkout/cart', ['_secure' => true]);
    }
}
