define([
    'underscore',
    'ko',
    'Magento_InventoryInStorePickupFrontend/js/model/pickup-locations-service',
    'Magento_Checkout/js/checkout-data',
    'Magento_Checkout/js/model/quote'
], function (_, ko, pickupLocationsService, checkoutData, quote) {
    'use strict';

    var mixin = {
        
        /**
         * Debug flag - set to true to enable debugging
         */
        debugEnabled: false,

        /**
         * Debug logging function
         */
        debugLog: function(message, data) {
            if (this.debugEnabled) {
                if (data !== undefined) {
                    console.log(message, data);
                } else {
                    console.log(message);
                }
            }
        },

        /**
         * Initialize - add debugging and inline store functionality
         */
        initialize: function () {
            self = this;
            // Initialize store selector functionality
            self.availableStores = ko.observableArray([]);
            self.inlineItemTemplate = 'Magento_InventoryInStorePickupFrontend/store-selector/inline-item';
            self.selectedLocationTemplate = 'Magento_InventoryInStorePickupFrontend/store-selector/selected-location';

            this._super();
            
            // Ensure selectedLocation is properly bound to the service
            if (!self.selectedLocation || typeof self.selectedLocation !== 'function') {
                self.selectedLocation = pickupLocationsService.selectedLocation;
            }

            self.nearbySearchRadius = 2000;
            self.updateNearbyLocations('NZ').then(function() {
                var locations = self.nearbyLocations();
                self.debugLog("nearbyLocations before sort", locations);

                // Enhance all locations with inventory status to ensure consistency
                locations = self.enhanceLocationsWithInventoryStatus(locations);

                locations = self.sortLocationsByAvailability(locations);
                            
                // Update the nearbyLocations observable with sorted data
                self.nearbyLocations(locations);
                self.availableStores(locations); 
                
                self.restorePersistedStoreSelection();

                // Validate currently selected location after loading stores
                self.validateSelectedLocationInventory();
            });
            return this;
        },

        openPopup: function() {
  
        },

        /**
         * Enhance locations with inventory status data
         */
        enhanceLocationsWithInventoryStatus: function(locations) {
            
            if (!locations || locations.length === 0) {
                return locations;
            }
            
            if (!window.checkoutConfig || !window.checkoutConfig.overdoseInStorePickup) {
                return locations;
            }
            
            // Enhance all locations with inventory data using the single location function
            var self = this;
            locations.forEach(function(location, index) {
                self.enhanceLocationWithInventory(location);
            });
            
            return locations;
        },

                /**
         * Enhance a single location with inventory status data
         */
        enhanceLocationWithInventory: function(location) {
            this.debugLog('🔍 enhanceLocationWithInventory called with:', location);
            
            if (!location) {
                return location;
            }
            
            if (!window.checkoutConfig || !window.checkoutConfig.overdoseInStorePickup) {
                return location;
            }
            
            var availableStores = window.checkoutConfig.overdoseInStorePickup.available_stores || [];
            var storesInventoryData = window.checkoutConfig.overdoseInStorePickup.stores_inventory_data || {};
            
            var sourceCode = location.pickup_location_code;
            var isAvailable = availableStores.indexOf(sourceCode) !== -1;
                        
            // Add inventory information
            if (storesInventoryData[sourceCode]) {
                location.inventory_status = storesInventoryData[sourceCode];
                location.status = isAvailable ? 'All items available for pickup' : 'Not all items available';
            } else {
                // No inventory data available for this store
                location.inventory_status = {
                    all_items_available: false,
                    items_status: {}
                };
                location.status = 'Not all items available';
            }
            
            return location;
        },
    

        /**
         * Find a store by pickup location code from nearby locations or create from pickup address
         */
        findStoreByLocationCode: function(storeCode) {
            this.debugLog('🔍 findStoreByLocationCode called with storeCode:', storeCode);
            
            // First try to find the store from nearby locations
            var nearbyLocations = this.nearbyLocations();
            var storeFromLocations = null;
            
            if (nearbyLocations && nearbyLocations.length > 0) {
                storeFromLocations = nearbyLocations.find(function(location) {
                    return location.pickup_location_code === storeCode;
                });
            }
            
            if (storeFromLocations) {
                this.debugLog('✅ Found store in nearby locations:', storeFromLocations);
                return storeFromLocations;
            }

            return null;
        },



        /**
         * Sort locations by availability
         * Available stores first, then unavailable
         */
        sortLocationsByAvailability: function(locations) {
            if (!locations || locations.length === 0) {
                return locations;
            }

            var sortedLocations = locations.slice();

            sortedLocations.sort(function(a, b) {

                // Define sorting priority: available = 1, unavailable = 2
                var getPriority = function(location) {
                    if (!location.inventory_status || !location.inventory_status.all_items_available) {
                        return 2; // Unavailable (including unknown)
                    }
                    return 1; // Available - highest priority
                };

                var priorityA = getPriority(a);
                var priorityB = getPriority(b);

                // Primary sort: by availability priority
                if (priorityA !== priorityB) {
                    return priorityA - priorityB;
                }

                return 0;
            });

            return sortedLocations;
        },

 

        /**
         * Validate that the currently selected location still has adequate inventory
         * If not, deselect it to prevent checkout with insufficient stock
         */
        validateSelectedLocationInventory: function() {
            var selectedLocation = pickupLocationsService.selectedLocation();
            
            this.debugLog("selectedLocation", selectedLocation);
            if (!selectedLocation) {
                return; // No location selected, nothing to validate
            }
            
            // Check if we have current inventory data
            if (!window.checkoutConfig || !window.checkoutConfig.overdoseInStorePickup) {
                return; // Can't validate without inventory data
            }
            
            var availableStores = window.checkoutConfig.overdoseInStorePickup.available_stores || [];
            var storesInventoryData = window.checkoutConfig.overdoseInStorePickup.stores_inventory_data || {};
            
            var storeCode = selectedLocation.pickup_location_code;
            
            // Check if the store is still available and has inventory
            var isCurrentlyAvailable = availableStores.indexOf(storeCode) !== -1;
            var hasInventoryData = storesInventoryData[storeCode];
            var hasAllItemsAvailable = hasInventoryData && storesInventoryData[storeCode].all_items_available === true;
            
            // If store is no longer available or doesn't have all items, deselect it
            if (!isCurrentlyAvailable || !hasAllItemsAvailable) {
                this.debugLog('Selected store no longer has adequate inventory, deselecting:', storeCode);
                
                // Clear the selection
                pickupLocationsService.selectForShipping(null);
                
                // Clear any persisted selection
                this.clearPersistedStoreData();
            }
        },

        // /**
        //  * Manually trigger validation of currently selected location
        //  * Useful when inventory data is updated dynamically
        //  */
        // checkSelectedLocationInventory: function() {
        //     this.validateSelectedLocationInventory();
        // },

        /**
         * Select a store option via radio input
         */
        selectStoreOption: function(location) {            
            // Only allow selection if store has inventory
            if (!location.inventory_status || !location.inventory_status.all_items_available) {
                this.debugLog('❌ Store does not have inventory, blocking selection');
                return false;
            }
            
            this.debugLog('✅ Store has inventory, proceeding with selection');
            // Persist the store selection for page refresh
            this.persistStoreSelection(location);            
             
            return true;
        },

        /**
         * Check if a store is currently selected
         */
        isStoreSelected: function(location) {
            var selectedPickupAddress = checkoutData.getSelectedPickupAddress();
            if (selectedPickupAddress && selectedPickupAddress.extension_attributes && selectedPickupAddress.extension_attributes.pickup_location_code) {
                return selectedPickupAddress.extension_attributes.pickup_location_code === location.pickup_location_code;
            }
            return false;
        },


        /**
         * Persist store selection using checkout data
         */
        persistStoreSelection: function(location) {
            try {
                this.debugLog('🔍 persistStoreSelection called with:', location);
                checkoutData.setSelectedPickupAddress(location);
                pickupLocationsService.selectForShipping(location);
                this.debugLog('✅ Store selection persisted via checkout data:', location.pickup_location_code);
            } catch (e) {
                console.error('❌ Error persisting store selection:', e);
            }
        },

        /**
         * Validate that a persisted store still has inventory for all current cart items
         */
        validatePersistedStoreInventory: function(persistedStore) {
            this.debugLog('🔍 validatePersistedStoreInventory called with:', persistedStore);
            
            // Check if we have current inventory data
            if (!window.checkoutConfig || !window.checkoutConfig.overdoseInStorePickup) {
                this.debugLog('❌ No checkout config or overdoseInStorePickup data available, allowing restoration');
                return true; // Allow restoration if we can't validate
            }
            
            var availableStores = window.checkoutConfig.overdoseInStorePickup.available_stores || [];
            var storesInventoryData = window.checkoutConfig.overdoseInStorePickup.stores_inventory_data || {};
            
            var storeCode = persistedStore.pickup_location_code;
            
            // Check if the store is in the current available stores list
            var isCurrentlyAvailable = availableStores.indexOf(storeCode) !== -1;
            
            if (!isCurrentlyAvailable) {
                this.debugLog('❌ Store is not currently available - clearing persisted store data');
                this.clearPersistedStoreData();
                return false;
            }
            this.debugLog('✅ Store is currently available');
            
            // Check if the store has inventory data and all items are available
            if (storesInventoryData[storeCode]) {
                var inventoryStatus = storesInventoryData[storeCode];
                var hasAllItemsAvailable = inventoryStatus.all_items_available === true;

                return hasAllItemsAvailable;
            } else {
                this.debugLog('❌ No inventory data for store - clearing persisted store data');
                this.clearPersistedStoreData();
                return false;
            }
        },
        /**
         * Check if the current shipping method is store pickup
         * This prevents restoring store selection when user is on delivery tab
         */
        isCurrentShippingMethodStorePickup: function() {
            this.debugLog('🔍 isCurrentShippingMethodStorePickup called');
            try {
                if (typeof checkoutData !== 'undefined' && checkoutData.getSelectedShippingRate) {
                    var selectedShippingRate = checkoutData.getSelectedShippingRate();
                    this.debugLog('🔍 Selected shipping rate from checkoutData:', selectedShippingRate);
                    if (selectedShippingRate && selectedShippingRate.indexOf('instore') !== -1) {
                        this.debugLog('✅ Found instore in selected shipping rate');
                        return true;
                    }
                }
                
                this.debugLog('❌ No store pickup shipping method found');
                return false;
            } catch (e) {
                console.error('❌ Error checking current shipping method:', e);
                return false;
            }
        },
                /**
         * Restore persisted store selection after page refresh
         */
        restorePersistedStoreSelection: function() {
            try {
                // Try to restore from checkoutData pickup address
                var selectedPickupAddress = checkoutData.getSelectedPickupAddress();
                
                if (!selectedPickupAddress) {
                    this.debugLog('❌ No selected pickup address found in checkoutData');
                    return;
                }
                
                
                 if (selectedPickupAddress.extension_attributes && selectedPickupAddress.extension_attributes.pickup_location_code) {
                     var storeCode = selectedPickupAddress.extension_attributes.pickup_location_code;
                     this.debugLog('✅ Found store code in pickup address:', storeCode);
                     
                     // Find the store using the reusable function
                     var storeToRestore = this.findStoreByLocationCode(storeCode);
                     
                     if (storeToRestore) {
                         this.debugLog('✅ Found store in nearby locations:', storeToRestore);
                     } else {
                         this.debugLog('❌ Store not found, clearing pickup address');
                         checkoutData.setSelectedPickupAddress(null);
                         return;
                     }
                    
                    // Validate that the persisted store still has inventory for all current cart items
                    var isValidInventory = this.validatePersistedStoreInventory(storeToRestore);
                    
                    if (isValidInventory) {
                        this.debugLog('✅ Store has valid inventory, proceeding with restoration');
                        // Use selectStoreOption to be consistent with user clicks
                        var self = this;
                        var isStorePickup = self.isCurrentShippingMethodStorePickup();                        
                        if (isStorePickup) {
                            this.debugLog('✅ Current shipping method is store pickup, calling selectStoreOption');
                            // Use selectStoreOption for consistent behavior
                            self.selectStoreOption(storeToRestore);
                        } else {
                            this.debugLog('❌ Current shipping method is not store pickup, clearing pickup address');
                            // Clear the selected pickup address if not on store pickup
                            checkoutData.setSelectedPickupAddress(null);
                        }
                        
                    } else {
                        this.debugLog('❌ Store does not have valid inventory, clearing pickup address');
                        // Clear the selected pickup address if store no longer has inventory
                        checkoutData.setSelectedPickupAddress(null);
                    }
                } else {
                    this.debugLog('❌ No pickup_location_code found in extension_attributes');
                }
                
            } catch (e) {
                console.error('❌ Error restoring persisted store selection:', e);
            }
        },


        /**
         * Clear persisted store data from checkout data
         */
        clearPersistedStoreData: function() {            
            try {
                if (checkoutData) {
                    checkoutData.setSelectedPickupAddress(null);
                    checkoutData.setSelectedShippingAddress(null);
                }
            } catch (e) {
                this.debugLog('Error clearing checkout data:', e);
            }
        }
    };

    return function (target) {
        return target.extend(mixin);
    };
});
